-- =============================================
-- Author:		Markp
-- Create date: 26/07/2013
-- Version:		1.0
-- Description:	Script to setup procedure to rebuild indexes and run it
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

PRINT 'Inserting tables...'

IF NOT EXISTS (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'[dbo].[QMS_PerformanceLog]') AND OBJECTPROPERTY(id,N'IsTable')=1)
BEGIN
	create table QMS_PerformanceLog(
	  ID              int            not null identity,
	  [Action]        nvarchar(1000),
	  StartTime       datetime,
	  EndTime         datetime,
	  RecordsAffected int,
	  DataSize        int
	)
END
GO


PRINT 'Creating Stored Procedures...'

IF NOT EXISTS (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'[dbo].[sys_Admin_SelectiveIndexMaintenance]') AND OBJECTPROPERTY(id,N'IsProcedure') = 1)
BEGIN
EXEC dbo.sp_executesql @statement = N'create PROCEDURE sys_Admin_SelectiveIndexMaintenance
AS
BEGIN
	SET NOCOUNT ON

	-- adapted from "Rebuild or reorganize indexes (with configuration)" from MSDN Books Online 
	-- (http://msdn.microsoft.com/en-us/library/ms188917.aspx)
	 
	-- =======================================================
	-- || Configuration variables:
	-- || - 10 is an arbitrary decision point at which to
	-- || reorganize indexes.
	-- || - 30 is an arbitrary decision point at which to
	-- || switch from reorganizing, to rebuilding.
	-- || - 0 is the default fill factor. Set this to a
	-- || a value from 1 to 99, if needed.
	-- =======================================================
	DECLARE @reorg_frag_thresh   float		SET @reorg_frag_thresh   = 5.0
	DECLARE @rebuild_frag_thresh float		SET @rebuild_frag_thresh = 10.0
	DECLARE @fill_factor         tinyint	SET @fill_factor         = 0
	-- added (DS) : page_count_thresh is used to check how many pages the current table uses
	DECLARE @page_count_thresh	 smallint	SET @page_count_thresh   = 1000
	
	-- Variables required for processing.
	DECLARE @objectid       int
	DECLARE @indexid        int
	DECLARE @partitioncount bigint
	DECLARE @schemaname     nvarchar(130) 
	DECLARE @objectname     nvarchar(130) 
	DECLARE @indexname      nvarchar(130) 
	DECLARE @partitionnum   bigint
	DECLARE @partitions     bigint
	DECLARE @frag           float
	DECLARE @page_count     int
	DECLARE @command        nvarchar(4000)
	DECLARE @intentions     nvarchar(4000)
	DECLARE @table_var      TABLE(
							  objectid     int,
							  indexid      int,
							  partitionnum int,
							  frag         float,
									  page_count   int
							)
							
	-- Variables for reporting
	DECLARE @job_start_date DATETIME
	DECLARE @command_start_date DATETIME
	SET @job_start_date = GETDATE()
	 
	-- Conditionally select tables and indexes from the
	-- sys.dm_db_index_physical_stats function and
	-- convert object and index IDs to names.
	INSERT INTO
		@table_var
	SELECT
		[object_id]                    AS objectid,
		[index_id]                     AS indexid,
		[partition_number]             AS partitionnum,
		[avg_fragmentation_in_percent] AS frag,
		[page_count]				   AS page_count
	FROM
		sys.dm_db_index_physical_stats (DB_ID(), NULL, NULL , NULL, ''LIMITED'')
	WHERE
		[avg_fragmentation_in_percent] > @reorg_frag_thresh 
		AND
		page_count > @page_count_thresh
		AND
		index_id > 0
	
 
	-- Declare the cursor for the list of partitions to be processed.
	DECLARE partitions CURSOR FOR
		SELECT * FROM @table_var
	 
	-- Open the cursor.
	OPEN partitions
 
	-- Loop through the partitions.
	WHILE (1=1) BEGIN
		FETCH NEXT
			FROM partitions
			INTO @objectid, @indexid, @partitionnum, @frag, @page_count
	 
		IF @@FETCH_STATUS < 0 BREAK
	 
		SELECT
			@objectname = QUOTENAME(o.[name]),
			@schemaname = QUOTENAME(s.[name])
		FROM
			sys.objects AS o WITH (NOLOCK)
			JOIN sys.schemas as s WITH (NOLOCK)
			ON s.[schema_id] = o.[schema_id]
		WHERE
			o.[object_id] = @objectid
	 
		SELECT
			@indexname = QUOTENAME([name])
		FROM
			sys.indexes WITH (NOLOCK)
		WHERE
			[object_id] = @objectid AND
			[index_id] = @indexid
	 
		SELECT
			@partitioncount = count (*)
		FROM
			sys.partitions WITH (NOLOCK)
		WHERE
			[object_id] = @objectid AND
			[index_id] = @indexid
	 
		-- Build the required statement dynamically based on options and index stats.
		SET @intentions =
			@schemaname + N''.'' +
			@objectname + N''.'' +
			@indexname + N'':'' + CHAR(13) + CHAR(10)
		SET @intentions =
			REPLACE(SPACE(LEN(@intentions)), '' '', ''='') + CHAR(13) + CHAR(10) +
			@intentions
		SET @intentions = @intentions +
			N'' FRAGMENTATION: '' + CAST(@frag AS nvarchar) + N''%'' + CHAR(13) + CHAR(10) +
			N'' PAGE COUNT: ''    + CAST(@page_count AS nvarchar) + CHAR(13) + CHAR(10)
	 
		IF @frag < @rebuild_frag_thresh BEGIN
			SET @intentions = @intentions +
				N'' OPERATION: REORGANIZE'' + CHAR(13) + CHAR(10)
			SET @command =
				N''ALTER INDEX '' + @indexname +
				N'' ON '' + @schemaname + N''.'' + @objectname +
				N'' REORGANIZE; '' + 
				N'' UPDATE STATISTICS '' + @schemaname + N''.'' + @objectname + 
				N'' '' + @indexname + '';''

		END
		IF @frag >= @rebuild_frag_thresh BEGIN
			SET @intentions = @intentions +
				N'' OPERATION: REBUILD'' + CHAR(13) + CHAR(10)
			SET @command =
				N''ALTER INDEX '' + @indexname +
				N'' ON '' + @schemaname + N''.'' +     @objectname +
				N'' REBUILD''
		END
		IF @partitioncount > 1 BEGIN
			SET @intentions = @intentions +
				N'' PARTITION: '' + CAST(@partitionnum AS nvarchar(10)) + CHAR(13) + CHAR(10)
			SET @command = @command +
				N'' PARTITION='' + CAST(@partitionnum AS nvarchar(10))
		END
		IF @frag >= @rebuild_frag_thresh AND @fill_factor > 0 AND @fill_factor < 100 BEGIN
			SET @intentions = @intentions +
				N'' FILL FACTOR: '' + CAST(@fill_factor AS nvarchar) + CHAR(13) + CHAR(10)
			SET @command = @command +
				N'' WITH (FILLFACTOR = '' + CAST(@fill_factor AS nvarchar) + '')''
		END
	 
		SET @command_start_date = GETDATE()
		
		SET @intentions = @intentions + N'' EXECUTING: '' + @command
		PRINT @intentions	    
		
		EXEC (@command)
		PRINT @command
		
		INSERT INTO dbo.QMS_PerformanceLog
		        ( Action ,
		          StartTime ,
		          EndTime ,
		          DataSize
		        )
		VALUES  ( @intentions , -- Action - nvarchar(1000)
		          @command_start_date, -- StartTime - datetime
		          GETDATE() , -- EndTime - datetime
		          @frag  -- DataSize - int
		        )
	END
	
	INSERT INTO dbo.QMS_PerformanceLog
	        ( Action ,
	          StartTime ,
	          EndTime ,
	          DataSize
	        )
	VALUES  ( N''Rebuild indexes job [sys_Admin_SelectiveIndexMaintenance] run'' , -- Action - nvarchar(1000)
	          @job_start_date , -- StartTime - datetime
	          GETDATE() , -- EndTime - datetime
	          @reorg_frag_thresh  -- DataSize - int
	        )
 
	-- Close and deallocate the cursor.
	CLOSE partitions
	DEALLOCATE partitions
END
'
END
GO



PRINT 'Removing archived attachments...'

IF NULLIF(object_id('tempdb.dbo.KeepingAssessment'), 0) IS NOT NULL				DROP TABLE tempdb.dbo.KeepingAssessment;
IF NULLIF(object_id('tempdb.dbo.KeepingAssessmentQuestion'), 0) IS NOT NULL		DROP TABLE tempdb.dbo.KeepingAssessmentQuestion;
IF NULLIF(object_id('tempdb.dbo.KeepingGuidance'), 0) IS NOT NULL				DROP TABLE tempdb.dbo.KeepingGuidance;
IF NULLIF(object_id('tempdb.dbo.DeletingAttachment'), 0) IS NOT NULL			DROP TABLE tempdb.dbo.DeletingAttachment;
IF NULLIF(object_id('tempdb.dbo.DeletingChunk'), 0) IS NOT NULL					DROP TABLE tempdb.dbo.DeletingChunk;
GO
CREATE TABLE tempdb.dbo.KeepingAssessment (AssessmentID int);
INSERT tempdb.dbo.KeepingAssessment(AssessmentID)
	SELECT AssessmentID FROM QMS_Assessment a
	WHERE a.AssessmentStatusID = 3
	AND a.RecordStatus < 201
GO
CREATE TABLE tempdb.dbo.KeepingAssessmentQuestion (AssessmentID int, AssessmentQuestionID int);
INSERT tempdb.dbo.KeepingAssessmentQuestion (AssessmentID, AssessmentQuestionID)
	SELECT qa.AssessmentID, qa.AssessmentQuestionID FROM QMS_AssessmentQuestion qa
	INNER JOIN tempdb.dbo.KeepingAssessment a ON qa.AssessmentID = a.AssessmentID
	WHERE qa.RecordStatus < 201
GO
CREATE TABLE tempdb.dbo.KeepingGuidance (GuidanceID int);
INSERT tempdb.dbo.KeepingGuidance (GuidanceID)
	SELECT GuidanceID FROM QMS_Guidance
	WHERE GuidanceID IN (
				SELECT GuidanceID FROM QMS_Assessment					WHERE AssessmentID IN (SELECT AssessmentID FROM tempdb.dbo.KeepingAssessment)
		UNION	SELECT GuidanceID FROM QMS_AssessmentQuestion			WHERE AssessmentID IN (SELECT AssessmentID FROM tempdb.dbo.KeepingAssessment)
		UNION	SELECT GuidanceID FROM QMS_AssessmentConstraint			WHERE AssessmentID IN (SELECT AssessmentID FROM tempdb.dbo.KeepingAssessment)
		UNION	SELECT GuidanceID FROM QMS_AssessmentQuestionConstraint	WHERE AssessmentQuestionID IN (SELECT AssessmentQuestionID FROM tempdb.dbo.KeepingAssessmentQuestion)
	)
	AND RecordStatus < 201;
GO
CREATE TABLE tempdb.dbo.DeletingAttachment (AttachmentID int, PageNumber int NULL);
INSERT tempdb.dbo.DeletingAttachment(AttachmentID, PageNumber)
	SELECT AttachmentID, NULL
	FROM QMS_Attachment a
	LEFT OUTER JOIN tempdb.dbo.KeepingGuidance g ON g.GuidanceID = a.SourceObjectID AND a.AttachmentSourceID = 3
	WHERE a.RecordStatus < 201
	AND a.AttachmentSourceID = 3
	AND g.GuidanceID IS NULL;
SELECT count(*) 'NumberOfExtraneousAttachments' FROM tempdb.dbo.DeletingAttachment
GO
DECLARE @PageSize int
SELECT @PageSize = count(*) / 10
FROM QMS_Attachment
WHERE AttachmentSourceID = 3;

IF NULLIF(object_id('tempdb.dbo.DeletingChunk'), 0) IS NOT NULL					DROP TABLE tempdb.dbo.DeletingChunk;
CREATE TABLE tempdb.dbo.DeletingChunk (PageNumber int, Number int, FromID int, ToID int);
INSERT tempdb.dbo.DeletingChunk(PageNumber, Number, FromID, ToID)
	SELECT PageNumber, count(*) 'Number', min(AttachmentID) 'From', max(AttachmentID) 'To'
	FROM 
	(	SELECT ROW_NUMBER() OVER (ORDER BY AttachmentID ASC) / @PageSize as PageNumber, AttachmentID
		FROM tempdb.dbo.DeletingAttachment
	) _
	GROUP BY PageNumber
	ORDER BY PageNumber ASC;
GO
UPDATE attachment
SET PageNumber = chunk.PageNumber
FROM tempdb.dbo.DeletingAttachment attachment
CROSS JOIN tempdb.dbo.DeletingChunk chunk
WHERE attachment.AttachmentID BETWEEN chunk.FromID and chunk.ToID
GO
EXEC SYNC_Log_Save 'I', 'Ready to remove non-active attachments'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 0;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 0'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 1;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 1'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 2;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 2'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 3;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 3'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 4;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 4'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 5;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 5'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 6;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 6'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 7;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 7'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 8;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 8'
GO
IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
	DELETE attachment
	FROM QMS_Attachment attachment
	INNER JOIN tempdb.dbo.DeletingAttachment deleting ON attachment.AttachmentID = deleting.AttachmentID AND deleting.PageNumber = 9;
EXEC SYNC_Log_Save 'I', 'Removed non-active attachments for page 9'
GO
DROP TABLE tempdb.dbo.KeepingAssessment;
DROP TABLE tempdb.dbo.KeepingAssessmentQuestion;
DROP TABLE tempdb.dbo.KeepingGuidance;
DROP TABLE tempdb.dbo.DeletingAttachment;
DROP TABLE tempdb.dbo.DeletingChunk;
EXEC SYNC_Log_Save 'I', 'Finished removing non-active attachments';
GO

--TODO
--remove archived assessment/data
--remove very old data (PM_ and check data)
--

PRINT 'Rebuilding indexes...'

EXEC	[dbo].[sys_Admin_SelectiveIndexMaintenance]
GO

PRINT 'Output results...'

SELECT     ID, Action, StartTime, EndTime, DataSize
FROM         QMS_PerformanceLog
WHERE     (EndTime > DATEADD(day, - 1, GETDATE()))
ORDER BY ID
GO

PRINT 'Indexes rebuilt successfully!'
GO

IF EXISTS (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'[dbo].[SP_RemoveOldMobileData]') AND OBJECTPROPERTY(id,N'IsProcedure') = 1)
BEGIN
	exec SP_RemoveOldMobileData 'muddy'
END
GO

DECLARE @RecoveryMode NVARCHAR(100)

SET @RecoveryMode =  CONVERT(NVARCHAR(100),(DATABASEPROPERTYEX('GreenlightQC_Mobile', 'Recovery')))

IF (@RecoveryMode = 'SIMPLE')
	BEGIN
   
		PRINT 'Recovery Mode Already Set to Simple' 
		PRINT 'Evaluating Log File Size' 

		DECLARE @totalLogSize BIGINT; 

		-- Lookup size of logfile
		-- JOIN/WHERE included to ensure only one row is returned for assignment
		SET @totalLogSize = (	SELECT du.total_log_size_in_bytes 
								FROM sys.dm_db_log_space_usage AS du 
								INNER JOIN sys.databases AS db
									ON du.database_id = db.database_id
								WHERE db.name = 'GreenlightQC_Mobile')

		IF (@totalLogSize IS NOT NULL)
		BEGIN
			-- If logfile exceeds 1gb, shrink it to address a legacy issue where the logfile had ballooned to several gigabytes (and align with tablet/phone space limitations) 
			IF (@totalLogSize > 1000000000) 
				BEGIN 
					DBCC SHRINKFILE(QMSOffice_log, 250, TRUNCATEONLY) -- Shrink logfile to 250mb, estimated right-sizing to avoid significant auto growth 
					PRINT 'Shrink Complete' 
				END 
			ELSE 
				PRINT 'Log File Size: ' +  CAST(@totalLogSize AS VARCHAR) + ' bytes' 
		END
		
	END
	ELSE
	BEGIN
		ALTER DATABASE GreenlightQC_Mobile
		SET RECOVERY SIMPLE
		DBCC SHRINKFILE (QMSOffice_log, 1)
		PRINT 'Recovery Mode Set to Simple'	
	END
