-- =============================================
-- Author:		Jeremy Formby
-- Create date: 3/05/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON

GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Device_List')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Device_List] AS SELECT ''stub version, to be replaced''')
GO
-- ===================================================
-- Author:		Jonathan Toolan
-- Create date:	13/5/2014
-- Description:	List all devices 
-- Version:		6
-- ===================================================
-- JT TFS 13-2931 - Created
-- MP TFS 4704: Update to make sure hidden versions still included
-- JT TFS 13-11262: Added column warning.
-- NM TFS 96065: Expanded to include devices running qc2/native-ios, added App & DeviceType columns, and updated order
-- NM TFS 144768/144717: Added Client ID, and updated version number ordering
-- ===================================================
ALTER PROCEDURE [dbo].[QRY_Device_List]
	@UserID int = NULL
AS
BEGIN
	SET NOCOUNT ON

		DECLARE @LatestVersion nvarchar(50)
		SELECT @LatestVersion = ApplicationVersion
		FROM APP_ApplicationUpdate
		WHERE ApplicationUpdateID = (	SELECT max(ApplicationUpdateID) 
										FROM APP_ApplicationUpdate
										WHERE Offline = 1 
										AND ApplicationEXE = 'Greenlight.QC.Mobile.exe')

		DECLARE @workstations TABLE (
			WorkStation nvarchar(50),
			LastLogin datetime NULL,
			AppVersion nvarchar(50) NULL
		)

		INSERT @workstations (WorkStation, AppVersion, LastLogin)
		SELECT WorkStation, AppVersion, LoggedIn
		FROM (
			SELECT	WorkStation, 
					AppVersion, 
					LoggedIn, 
					row_number() OVER (PARTITION BY WorkStation ORDER BY LoggedIn DESC) as row
			FROM QMS_UserLog
			WHERE AppName = 'Greenlight QC Mobile'
		) AS workstations
		WHERE row = 1

	SET NOCOUNT OFF

	-- WARNING: If you change the columns type or order, please update APP_Notification_OutOfDateDevices
	SELECT	list.ClientGUID,
			list.ClientID,
			CASE WHEN list.[Enabled] = 0				THEN 'Disabled'
				 WHEN work.AppVersion = @LatestVersion	THEN 'Up-to-date'
				 WHEN list.Application IS NOT NULL		THEN 'N/A'
														ELSE 'Update required' END StatusText,
			CASE WHEN list.[Enabled] = 0				THEN 180
				 WHEN work.AppVersion = @LatestVersion	THEN 20
				 WHEN list.Application IS NOT NULL		THEN null
														ELSE 2 END StatusID,
			list.MachineName,
			coalesce(work.AppVersion, list.Version, list.iosNativeVersion, null) as AppVersion,	-- Use windows native version if available, QC2, ios-native, or null 
			list.App,
			list.DeviceType,
			list.LastSync,
			list.[Enabled]			
	FROM (
		SELECT	ClientGUID,
				ClientID,
				MachineName,
				WorkStation = substring(MachineName, CASE WHEN MachineName LIKE '%\%' THEN charindex('\', MachineName) + 1 ELSE 0 END, 50),
				LastSync,
				Application,
				Version,

				CASE WHEN charindex('GreenlightQC', Application) > 0	THEN 'QC2'
				WHEN charindex('QualityControl', Application) > 0		THEN 'Native iOS'
																		ELSE 'Mobile for Windows' END App,

								   -- check for presence of qualifying characters, and that substring can't generate a negative number
				iosNativeVersion = iif((charindex('QualityControl', Application) > 0) AND (charindex(' ', Application) > 0) AND (charindex('/', Application) > 0)
								   AND (charindex(' ', Application) - charindex('/', Application) > 0),

										substring(Application,	-- fetch version located between '/' and ' ' 
											charindex('/', Application) + 1, 
											charindex(' ', Application) - charindex('/', Application)), 

								   null),	-- where we didn't pattern match a version, return null

							 -- check for presence of qualifying characters, and that substring can't generate a negative number
				DeviceType = iif((charindex('GreenlightQC', Application) > 0) AND (charindex('|', Application) > 0) 
							 AND (charindex('|', Application, (charindex('|', Application))+1) > 0), 

								substring(Application,	-- fetch version located between first '|' and second '|' 
									charindex('|', Application) + 1, 
									(charindex('|', Application, (charindex('|', Application))+1) - charindex('|', Application)-1)), 

							 null),	-- where we didn't pattern match a device type, return null

				[Enabled]
		FROM SYNC_Client
		WHERE ClientType IN (1, 4, 5, 8, 9)	-- Check windows, ios and android
	) AS list
	LEFT OUTER JOIN @workstations work ON list.WorkStation = work.WorkStation
	ORDER BY App, case when DeviceType is null then 2 else 1 end, DeviceType, LastLogin DESC, MachineName
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Notification_OutOfDateDevices')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Notification_OutOfDateDevices] AS SELECT ''stub version, to be replaced''')
GO
-- =====================================================================
-- Author:		Jonathan Toolan
-- Create date: 20/11/2014
-- Description:	Sends emails to admins when devices are out of date.
-- Version:		7
-- =====================================================================
-- JT TFS 13-11262 - Created with link to QRY_Device_List
-- JT TFS 13-11657 - Alert will not go out if ApplicationVersion is not specified in APP_ApplicationUpdate
-- JT TFS 13-11262 - Changed text of email.
-- JT TFS 13-11657 - Do not send out emails for machines for which we do not know the version.
-- JT TFS 13-11657 - Do not send out emails if the messagebody is null (for any reason)
-- NM TFS 96065	   - Match structure updates to QRY_Device_List
-- NM TFS 144768   - Added ClientID
-- =====================================================================
ALTER PROCEDURE [dbo].[APP_Notification_OutOfDateDevices]
AS
BEGIN

	-- Latest Version
	DECLARE @LatestVersion nvarchar(50)
	SELECT @LatestVersion = ApplicationVersion
	FROM APP_ApplicationUpdate
	WHERE ApplicationUpdateID = (	SELECT max(ApplicationUpdateID) 
									FROM APP_ApplicationUpdate
									WHERE Offline = 1 
									AND ApplicationEXE = 'Greenlight.QC.Mobile.exe')

	--Link device log screen
	DECLARE @BaseURL nvarchar(1000)
	SELECT @BaseURL = dbo.FNC_GetOptionValue(65)

	DECLARE @LogDeviceURL nvarchar(max)
	SELECT @LogDeviceURL =  @BaseURL  + CASE WHEN @BaseURL LIKE '%/' THEN '' ELSE '/' END + MenuAction 
	FROM APP_Menu 
	WHERE ResourceName = 'MNU_MobileDeviceLog' 
	AND AppID = 3

	-- Which devices need updating?
	DECLARE @Devices TABLE (
		ClientGUID uniqueidentifier,
		ClientID int,
		StatusText nvarchar(50),
		StatusID int,
		MachineName nvarchar(50),
		AppVersion nvarchar(50),
		App nvarchar(50),
		DeviceType nvarchar(50),
		LastSync datetime,
		[Enabled] bit
	)
	INSERT @Devices EXEC QRY_Device_List 1



	-- Build message body in two steps.
	DECLARE @DeviceList nvarchar(max)
	SELECT @DeviceList = substring(
				(	SELECT nchar(13) + nchar(10) + MachineName + N' (v' + AppVersion + N')'
					FROM @Devices WHERE StatusID = 2
					AND AppVersion IS NOT NULL
					AND MachineName IS NOT NULL
					FOR XML PATH (''), type
				).value('.', 'nvarchar(max)'),
			3,
			2147483647)

	--If there no devices needing an update, no email is sent
	IF NULLIF(@DeviceList, '') IS NULL
	BEGIN
		DECLARE @MessageBody nvarchar(max)
SELECT @MessageBody = 
N'The latest version of the mobile device software is ' + ISNULL(@LatestVersion, 'available.') + N'

The following mobile devices are out of date, and require an update:

' + @DeviceList + N'

Please visit ' + coalesce(@LogDeviceURL, @BaseURL, 'your Greenlight QC website')  + ' to download the latest version to install on those devices.

This is an automatically created report for your information. You will only receive this report if you have any devices that still need updating.'

		-- Send to all users in new IT support group
		DECLARE @Admins TABLE (Fullname nvarchar(255), Email nvarchar(100))
		INSERT @Admins EXEC QRY_User_ListOfAdmins

		--Enter QMS_Notification record listing all devices that are out of date, including Device name, device version
		INSERT QMS_Notification(SendTo, Subject, MessageBody)
		SELECT	[sendto] = admins.Email,
				[subject] = 'Mobile devices which require an update.',
				[message] = @MessageBody
		FROM @Admins admins
		WHERE @MessageBody IS NOT NULL
	END
END
GO



IF EXISTS (SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = 'dbo' AND TABLE_NAME = 'QMS_CorrectiveAction' AND COLUMN_NAME = 'CASubTitle' AND CHARACTER_MAXIMUM_LENGTH != 255)
BEGIN
	ALTER TABLE QMS_CorrectiveAction
	ALTER COLUMN CASubTitle nvarchar(255) NULL	
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_CorrectiveAction')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_CorrectiveAction] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 01/08/2008
-- Description:	Saves corrective action
-- Version:		18
-- =============================================
-- j.toolan 21 Feb 2012: Added ReminderDate
-- j.toolan  7 Mar 2012: Reset NotifcationSent on 6 if AssignSupplierID set (only if status changes)
-- MP TFS 5873 - Made sure SourceObjectID updated so that correct links to response created
-- JE TFS 7087 - added CorrectiveActionTypeID to UPDATE statement
-- JE TFS 7877 - corrected resetting of NotifcationSent for Completed alerts
-- JE TFS 7836 - updated to save LocationID CA is assigned
-- JT TFS 13-7950 - Make sure that if CorrectiveActionTypeID is specified, and CorrectiveActionEscalationID is NOT, then set it.
-- JE TFS 13-12635: call new SAVE_CorrectiveActionUser SP to save permissions data to QMS_CorrectiveActionUser table
-- MP TFS 13-15377: excluded call to new SAVE_CorrectiveActionUser SP for mobile
-- AC TFS 6335: Changes the escalation if the corrective action type or notification user group is changed
-- PH TFS 13-19579: 06/05/2015 - Made compatible with WebChecks service/EF.  @CorrectiveActionID must have default (null or zero) because stripped out by EF when adding record and @CorrectiveActionID must be returned AS CorrectiveActionID.
-- PH TFS 13-21031: 10/06/2015 - Got rid of code smell that may have been causing bug.  New CorrectiveActionID obtained OUTPUT INSERTED.CorrectiveActionID INTO @caIdTable
-- PH TFS 13-23902: 15/07/2015 - SourceObjectID is nullable, so parameter should not be mandatory when saving an Ad Hoc corrective action
-- JT TFS 13-59936: 25/04/2017 - Allow 100 characters for CATitle (to reflect underlying table)
-- TS TFS 13-88643: 05/06/18 - Capture UTCOFfset
-- TS TFS 13-114143: 14/03/19 - Increase imit of CASubTitle 
-- =============================================
ALTER PROCEDURE [dbo].[SAVE_CorrectiveAction]
	@CorrectiveActionID int = NULL,
	@CASourceID int,
	@SourceObjectID int = NULL,
	@CATitle nvarchar(100),
	@CASubTitle nvarchar(255),
	@CAReason nvarchar(1000),
	@CARequired nvarchar(1000),
	@CAStatusID int,
	@CorrectiveActionTypeID int,
	@CorrectiveActionEscalationID int,
	@NotificationGroupID int,
	@AssignUserID int,
	@DateOccurred datetime,
	@PriorityLevelID int,
	@DateLogged datetime,
	@DateDeadline datetime,
	@ActionTaken nvarchar(1000),
	@Comments nvarchar(1000),
	@DateCompleted datetime,
	@Completed_UserID int,
	@Completed_MachineName nvarchar(50),
	@DateSignedOff datetime,
	@SignOff_UserID int,
	@SignOff_MachineName nvarchar(50),
	@UserID int,
	@AssignSupplierID int,
	@ReminderDate datetime,
	@ReminderSent bit = NULL, -- This is not saved currently.
	@LocationID int
AS
BEGIN
	SET NOCOUNT ON

	-- BUG 7950 SP FIX
	SELECT @LocationID = NULLIF(@LocationID, 0) -- Do not allow zero for LocationID. It stops first new CA alert going out.

	DECLARE @DefaultCorrectiveActionEscalationID int
	SELECT @DefaultCorrectiveActionEscalationID = CorrectiveActionEscalationID 
	FROM QMS_CorrectiveActionEscalation 
	WHERE CorrectiveActionTypeID = @CorrectiveActionTypeID
	AND ListIndex = (	SELECT min(ListIndex) FROM QMS_CorrectiveActionEscalation 
						WHERE CorrectiveActionTypeID = @CorrectiveActionTypeID)
	-- END BUG 7950 fix

	IF ISNULL(@CorrectiveActionID,0)=0 --New record
	BEGIN
		DECLARE @caIdTable TABLE (CorrectiveActionID int)

		DECLARE @UTCOffset FLOAT
		SET @UTCOffset = DATEDIFF(mi, GETUTCDATE(), GETDATE()) / 60

		INSERT INTO QMS_CorrectiveAction
						  (CASourceID, SourceObjectID, CATitle, CASubTitle, CAReason, CARequired, CAStatusID, CorrectiveActionTypeID, CorrectiveActionEscalationID, NotificationGroupID, 
						  AssignUserID, DateOccurred, PriorityLevelID, DateLogged, DateDeadline, Logged_UserID, Logged_MachineName, ActionTaken, Comments, DateCompleted, 
						  Completed_UserID, Completed_MachineName, AssignSupplierID, ReminderDate, LocationID, UTCOffset)
				OUTPUT INSERTED.CorrectiveActionID INTO @caIdTable
		VALUES     (@CASourceID,@SourceObjectID,@CATitle,@CASubTitle,@CAReason,@CARequired,@CAStatusID,@CorrectiveActionTypeID,
					ISNULL(@CorrectiveActionEscalationID, @DefaultCorrectiveActionEscalationID),
					@NotificationGroupID,@AssignUserID,@DateOccurred,@PriorityLevelID,@DateLogged,@DateDeadline,@UserID,
						   HOST_NAME(),@ActionTaken,@Comments,@DateCompleted,@Completed_UserID,@Completed_MachineName,@AssignSupplierID, @ReminderDate, @LocationID, @UTCOffset)
	                      
		--Get new ID
		SELECT     @CorrectiveActionID = MAX(CorrectiveActionID)
		FROM         @caIdTable
	END
	ELSE --Update record
	BEGIN
		DECLARE @originalCAStatusID int
		SELECT	@originalCAStatusID = CAStatusID 
		FROM	QMS_CorrectiveAction
		WHERE	CorrectiveActionID = @CorrectiveActionID
		
		DECLARE @originalCorrectiveActionTypeID int
		DECLARE @originalNotificationGroupID int
		
		SELECT	@originalCorrectiveActionTypeID = CorrectiveActionTypeID, @originalNotificationGroupID = NotificationGroupID
		FROM	QMS_CorrectiveAction
		WHERE	CorrectiveActionID = @CorrectiveActionID
		
		-- Only update the escalation if the corrective action type or the notification user group has changed
		IF (@originalCorrectiveActionTypeID <> @CorrectiveActionTypeID OR @originalNotificationGroupID <> @NotificationGroupID)
		BEGIN
			SET @CorrectiveActionEscalationID = NULL
			SELECT @CorrectiveActionEscalationID = CorrectiveActionEscalationID
			FROM QMS_CorrectiveActionEscalation
			WHERE CorrectiveActionTypeId = @CorrectiveActionTypeId
			AND NotificationGroupId = @NotificationGroupId
		END

		UPDATE    QMS_CorrectiveAction
		SET              SourceObjectID = @SourceObjectID, CATitle = @CATitle, CASubTitle = @CASubTitle, CAReason = @CAReason, CARequired = @CARequired, 
							  CAStatusID = @CAStatusID, CorrectiveActionTypeID = @CorrectiveActionTypeID, NotificationGroupID = @NotificationGroupID, AssignUserID = @AssignUserID, DateOccurred = @DateOccurred, 
							  PriorityLevelID = @PriorityLevelID, DateDeadline = @DateDeadline, ActionTaken = @ActionTaken, Comments = @Comments, DateCompleted = @DateCompleted, 
							  Completed_UserID = @Completed_UserID, Completed_MachineName = @Completed_MachineName, DateSignedOff = @DateSignedOff, 
							  SignOff_UserID = @SignOff_UserID, SignOff_MachineName = @SignOff_MachineName, 
							  RecordStatus = CASE WHEN RecordStatus = 30 THEN 20 ELSE RecordStatus END, Updated = GETUTCDATE(), AssignSupplierID = @AssignSupplierID, 
							  ReminderDate = @ReminderDate, LocationID = @LocationID, CorrectiveActionEscalationID = @CorrectiveActionEscalationID
		WHERE     (CorrectiveActionID = @CorrectiveActionID)

		-- Only update notification flag if the status has changed.
		IF (@originalCAStatusID <> @CAStatusID)
		BEGIN
				--get Failure_NotificationGroupID for current CA escalation
			DECLARE @Failure_NotificationGroupID int 
			SELECT     @Failure_NotificationGroupID = Failure_NotificationGroupID
			FROM         QMS_CorrectiveActionEscalation
			WHERE     (CorrectiveActionEscalationID = @CorrectiveActionEscalationID)

			--if CA being set to Complete and a Failure_NotificationGroup has been set, reset NotificationGroup flag
			IF @CAStatusID = 4 --AND @Failure_NotificationGroupID IS NOT NULL --removed for TFS 7877 as this stops Completed alerts firing when no Failure_NotificationGroupID set
			BEGIN
				UPDATE	QMS_CorrectiveAction
				SET		NotificationSent = 0
				WHERE	(CorrectiveActionID = @CorrectiveActionID)
			END
			
				--if CA being set to Signed-off and AssignSupplierID has been set, reset NotificationGroup flag
			IF @CAStatusID IN (5, 6) AND @AssignSupplierID IS NOT NULL
			BEGIN
				UPDATE	QMS_CorrectiveAction
				SET		NotificationSent = 0
				WHERE	(CorrectiveActionID = @CorrectiveActionID)
			END
		END
	END
	SET NOCOUNT OFF

	--save data for CA to QMS_CorrectiveActionUser (should handle both new and updated CAs as @CorrectiveActionID is set either way)
    IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
        BEGIN --Add current user to current corrective action
            IF NOT EXISTS ( SELECT  CorrectiveActionID
                            FROM    dbo.QMS_CorrectiveActionUser
                            WHERE   UserID = @UserID
                                    AND CorrectiveActionID = @CorrectiveActionID )
                INSERT  INTO dbo.QMS_CorrectiveActionUser
                        ( CorrectiveActionID, UserID )
                VALUES  ( @CorrectiveActionID, -- CorrectiveActionID - int
                          @UserID -- UserID - int
                          )
        END
    ELSE
        EXEC SAVE_CorrectiveActionUser @CorrectiveActionID
	

	SELECT @CorrectiveActionID AS CorrectiveActionID
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SYNC_Import_CorrectiveAction')
	EXEC ('CREATE PROCEDURE [dbo].[SYNC_Import_CorrectiveAction] AS SELECT ''stub version, to be replaced''')
GO
-- ===================================================
-- Author:		Jonathan Toolan
-- Create date:	5 Jun 2018
-- Description:	[insert description here]
-- Version:		2
-- ===================================================
-- NM TFS 133429 - Updated corrective action size
-- ===================================================
ALTER PROCEDURE [dbo].[SYNC_Import_CorrectiveAction] 
	@CASourceID int,
	@SourceObject_GUID uniqueidentifier,
	@CATitle nvarchar(50),
	@CAReason nvarchar(1000),
	@CARequired nvarchar(1000),
	@CAStatusID int,
	@CorrectiveActionTypeID int,
	@CorrectiveActionEscalationID int,
	@NotificationGroupID int,
	@PriorityLevelID int,
	@DateLogged datetime,
	@DateOccurred datetime = NULL,
	@DateDeadline datetime,
	@Logged_UserID int,
	@Logged_MachineName nvarchar(50),
	@NotificationSent bit,
	@ActionTaken nvarchar(1000),
	@Comments nvarchar(1000),
	@DateCompleted datetime,
	@Completed_UserID int,
	@Completed_MachineName nvarchar(50),
	@DateSignedOff datetime,
	@SignOff_UserID int,
	@SignOff_MachineName nvarchar(50),
	@DateFailed datetime,
	@CASubTitle nvarchar(255),
	@AssignUserID int,
	@AssignSupplierID int = NULL,
	@ReminderDate datetime = NULL,
	@ReminderSent bit = NULL,
	@LocationID INT = NULL,
	@UTCOffset float = NULL,
	@RecordStatus tinyint,
	@Inserted datetime,
	@Updated datetime,
	@Deleted datetime,
	@GUID uniqueidentifier,
	@Created bigint,
	@RowVersion timestamp,
	@InsertCount int = 0 OUTPUT,
	@UpdateCount int = 0 OUTPUT,
	@ReceivedAnchor bigint = 0 OUTPUT
AS

--Lookup SourceObjectID from GUID
DECLARE @SourceObjectID int
IF @CASourceID = 1 --Response
	SELECT @SourceObjectID = ResponseID FROM QMS_Response WHERE (GUID = @SourceObject_GUID)
IF @CASourceID = 2 --ResponseQuestion
	SELECT @SourceObjectID = ResponseQuestionID FROM QMS_ResponseQuestion WHERE (GUID = @SourceObject_GUID)
IF @CASourceID = 3 --CorrectiveAction
	SELECT @SourceObjectID = CorrectiveActionID FROM QMS_CorrectiveAction WHERE (GUID = @SourceObject_GUID)

IF @ReminderSent IS NULL
	SET @ReminderSent = 0

IF NOT EXISTS(SELECT * FROM QMS_CorrectiveAction WHERE (GUID = @GUID))
BEGIN --Append record
	IF @RecordStatus < 201 AND @CAStatusID < 3 --Dont append closed actions
	BEGIN
		INSERT INTO QMS_CorrectiveAction
                      (CASourceID, SourceObjectID, CATitle, CAReason, CARequired, CAStatusID, CorrectiveActionTypeID, CorrectiveActionEscalationID, NotificationGroupID, PriorityLevelID, 
                      DateLogged, DateOccurred, DateDeadline, Logged_UserID, Logged_MachineName, NotificationSent, ActionTaken, Comments, DateCompleted, Completed_UserID, 
                      Completed_MachineName, DateSignedOff, SignOff_UserID, SignOff_MachineName, DateFailed, CASubTitle, AssignUserID, AssignSupplierID, ReminderDate, 
                      ReminderSent, LocationID, UTCOffset, RecordStatus, Inserted, Updated, Deleted, GUID)
		VALUES     (@CASourceID,@SourceObjectID,@CATitle,@CAReason,@CARequired,@CAStatusID,@CorrectiveActionTypeID,@CorrectiveActionEscalationID,@NotificationGroupID,@PriorityLevelID,@DateLogged,@DateOccurred,@DateDeadline,@Logged_UserID,@Logged_MachineName,@NotificationSent,@ActionTaken,@Comments,@DateCompleted,@Completed_UserID,@Completed_MachineName,@DateSignedOff,@SignOff_UserID,@SignOff_MachineName,@DateFailed,@CASubTitle,@AssignUserID,@AssignSupplierID,@ReminderDate,@ReminderSent,@LocationID,
                       @UTCOffset, 30,@Inserted,@Updated,@Deleted,@GUID)
		SET @InsertCount = @@rowcount
	END
END

ELSE
BEGIN --Update record
	UPDATE    QMS_CorrectiveAction
	SET              CATitle = @CATitle, CAReason = @CAReason, CARequired = @CARequired, CAStatusID = @CAStatusID, CorrectiveActionTypeID = @CorrectiveActionTypeID, 
                      CorrectiveActionEscalationID = @CorrectiveActionEscalationID, NotificationGroupID = @NotificationGroupID, PriorityLevelID = @PriorityLevelID, 
                      DateLogged = @DateLogged, DateOccurred = @DateOccurred, DateDeadline = @DateDeadline, Logged_UserID = @Logged_UserID, 
                      Logged_MachineName = @Logged_MachineName, NotificationSent = @NotificationSent, ActionTaken = @ActionTaken, Comments = @Comments, 
                      DateCompleted = @DateCompleted, Completed_UserID = @Completed_UserID, Completed_MachineName = @Completed_MachineName, 
                      DateSignedOff = @DateSignedOff, SignOff_UserID = @SignOff_UserID, SignOff_MachineName = @SignOff_MachineName, DateFailed = @DateFailed, 
                      CASubTitle = @CASubTitle, AssignUserID = @AssignUserID, AssignSupplierID = @AssignSupplierID, ReminderDate = @ReminderDate, 
                      ReminderSent = @ReminderSent, RecordStatus = CASE WHEN @RecordStatus < 30 THEN 30 ELSE @RecordStatus END, LocationID = @LocationID, 
                      UTCOffset = @UTCOffset, Inserted = @Inserted, Updated = @Updated, Deleted = @Deleted
	WHERE     (GUID = @GUID)
	SET @UpdateCount = @@rowcount
END

IF CAST(@RowVersion AS bigint) > @Created
	SET @ReceivedAnchor = CAST(@RowVersion AS bigint)
ELSE
	SET @ReceivedAnchor = @Created
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_Response_MultiCheckQuestion')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_Response_MultiCheckQuestion] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		: Jon Edwards
-- Create date	: 10/01/2014
-- Description	: Returns data for Multi Check parent and child questions
-- Version		: 8
-- =============================================
-- JE TFS 8717		: amended filter to show archived questions if they have a response against them
-- JE TFS 13-1588	: added RecordStatus column
-- JE TFS 13-2555	: display Score after response when Score > 0 and return NoOfQuestionsToHide
-- JE TFS 13-43069	: Changed functionality when applying 'Suppress Blanks' setting.  The process now excludes questions 
--                    where there are null responses, but does not exclude null responses from included questions. The 
--                    score and qa flags are duplicated in every question response of a check and exlcusion may result in
--                    the score/qa flag being shown on the report.
-- JT TFS 125510	: Make sure response data is shown even if the related assessment is deleted or archived (by recordstatus).
-- NM TFS 83743		: Make sure child checks are correctly ordered in association with their parent check.
-- =============================================
ALTER PROCEDURE [dbo].[RPT_Response_MultiCheckQuestion]
	@UserID int
AS
BEGIN

--declare @UserID int
--select @UserID = 1

	DECLARE @SupressBlanks bit;
	SET @SupressBlanks = ISNULL(CAST(dbo.FNC_GetOptionValue(36) AS bit),0);

	IF NULLIF(object_id('tempdb..#tmpAllAssesmentQuestions'), 0) IS NOT NULL DROP TABLE #tmpAllAssesmentQuestions
	SELECT
		question.AssessmentID,
		sort.AssessmentQuestionID,
		sort.SortIndex,
		sort.QuestionNumber
	INTO #tmpAllAssesmentQuestions
	FROM dbo.FNC_AssessmentQuestion_Sort_Multi(@UserID) sort -- This is the most expensive bit of code.
	INNER JOIN QMS_AssessmentQuestion question
		ON sort.AssessmentQuestionID = question.AssessmentQuestionID;

	IF NULLIF(object_id('tempdb..#tmpShownAssessmentQuestions'), 0) IS NOT NULL DROP TABLE #tmpShownAssessmentQuestions
	SELECT responseQuestion.ResponseID, includedAssessmentQuestions.AssessmentQuestionID 
	INTO #tmpShownAssessmentQuestions
	FROM
	(
		SELECT assessmentQuestions.AssessmentQuestionID
		FROM #tmpAllAssesmentQuestions AS assessmentQuestions
		INNER JOIN QMS_ResponseQuestion
			ON QMS_ResponseQuestion.AssessmentQuestionID = assessmentQuestions.AssessmentQuestionID
		INNER JOIN TEMP_Selected
			ON QMS_ResponseQuestion.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID
		INNER JOIN QMS_Response AS response
			ON response.ResponseID = TEMP_Selected.SelectedID
		WHERE
			response.ParentResponseGUID IS NOT NULL
			AND
			(
				(
					@SupressBlanks = 1
					AND
					(
						QMS_ResponseQuestion.Response_Text IS NOT NULL
						OR LEN(LTRIM(RTRIM(QMS_ResponseQuestion.Response_Text))) > 0
					)
				)
				OR
				(
					@SupressBlanks = 0
				)
			)
		GROUP BY assessmentQuestions.AssessmentQuestionID
	) AS includedAssessmentQuestions
	INNER JOIN QMS_ResponseQuestion AS responseQuestion
		ON responseQuestion.AssessmentQuestionID = includedAssessmentQuestions.AssessmentQuestionID
	INNER JOIN TEMP_Selected
		ON responseQuestion.ResponseID = TEMP_Selected.SelectedID
		AND TEMP_Selected.UserID = @UserID;

	-- Add headers for shown items if suppressing blanks.
	IF (@SupressBlanks = 1)
		INSERT #tmpShownAssessmentQuestions
		(
			ResponseID,
			AssessmentQuestionID
		)
		SELECT DISTINCT
			shown.ResponseID,
			questionChildren.AssessmentQuestionID
		FROM
		(
			SELECT	questions.AssessmentQuestionID,
			(
				SELECT '|' + convert(varchar(10), AssessmentQuestionID
			)
			FROM #tmpAllAssesmentQuestions
			WHERE
				QuestionNumber LIKE questions.QuestionNumber + '.%'
				AND AssessmentID = questions.AssessmentID FOR XML PATH('')) + '|' Children
			FROM #tmpAllAssesmentQuestions questions
		) AS questionChildren
		INNER JOIN #tmpShownAssessmentQuestions AS shown
			ON questionChildren.Children like '%|' + convert(varchar(10), shown.AssessmentQuestionID) + '|%' 
		WHERE
			NOT EXISTS
			(
				SELECT *
				FROM #tmpShownAssessmentQuestions
				WHERE
					ResponseID = shown.ResponseID
					AND AssessmentQuestionID = questionChildren.AssessmentQuestionID
			);

	--get Question data for parent and child checks
	SELECT
		QMS_ResponseQuestion.ResponseID,
		QMS_AssessmentQuestion.AssessmentQuestionID,
		AssessmentQuestion_Sort.QuestionNumber,
		QMS_QuestionPool.Question,
		APP_QuestionLevel.QuestionLevel,
		APP_QuestionLevel.ListIndex AS QuestionLevelSort,
		QMS_ResponseQuestion.Response_Text +
						CASE WHEN ISNULL(QMS_ResponseQuestion.Score, 0) = 0 THEN ''
								ELSE ' (' + CAST(QMS_ResponseQuestion.Score AS varchar(5)) + ')' END AS Response_Text, --include score after response when it is > 0
		QMS_ResponseQuestion.Recommended_QAFlagID,
		QMS_ResponseQuestion.QuestionComments,
		QMS_QuestionPool_Parent.Question AS ParentQuestion,
		APP_QAFlag.QAFlagBackColour,
		APP_QAFlag.QAFlagForeColour,
		QMS_ResponseQuestion.ResponseQuestionID,
		IsSectionHeaderRow = convert(bit, CASE WHEN NULLIF(QMS_AssessmentQuestion.DataTypeID, 0) IS NULL THEN 1 ELSE 0 END),
		TEMP_Selected.SelectedValue, --this indicates if question row is belongs to a parent or child check
		QMS_Response.Actual_Score,
		OverallFlag.QAFlag AS OverallFlag,
		OverallFlag.QAFlagBackColour AS OverallBackColour,
		QMS_Response.RecordStatus,
		--questions are in blocks of 5, the below calculates how many we need to hide
		CASE WHEN
		(
			ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) < NumberOfChildChecks.NoOfChildChecks
			THEN (((ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) + 5) - NumberOfChildChecks.NoOfChildChecks)
			ELSE ((ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) - NumberOfChildChecks.NoOfChildChecks
		) END AS NoOfQuestionsToHide
	FROM	TEMP_Selected
	INNER JOIN QMS_Response
		ON TEMP_Selected.SelectedID = QMS_Response.ResponseID
	INNER JOIN APP_QAFlag AS OverallFlag
		ON QMS_Response.Actual_QAFlagID = OverallFlag.QAFlagID
	INNER JOIN QMS_ResponseQuestion
		ON TEMP_Selected.SelectedID = QMS_ResponseQuestion.ResponseID
	INNER JOIN QMS_AssessmentQuestion
		ON QMS_ResponseQuestion.AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID
	INNER JOIN #tmpShownAssessmentQuestions AS visibleQuestions
		ON visibleQuestions.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID
	INNER JOIN #tmpAllAssesmentQuestions AS AssessmentQuestion_Sort
		ON visibleQuestions.AssessmentQuestionID = AssessmentQuestion_Sort.AssessmentQuestionID
	INNER JOIN QMS_QuestionPool
		ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID
	LEFT OUTER JOIN APP_QAFlag
		ON QMS_ResponseQuestion.Recommended_QAFlagID = APP_QAFlag.QAFlagID
	LEFT OUTER JOIN QMS_AssessmentQuestion AS QMS_AssessmentQuestion_Parent
		ON QMS_AssessmentQuestion.ParentQuestionID = QMS_AssessmentQuestion_Parent.AssessmentQuestionID
	LEFT OUTER JOIN APP_QuestionLevel
		ON QMS_AssessmentQuestion.QuestionLevelID = APP_QuestionLevel.QuestionLevelID
	LEFT OUTER JOIN QMS_QuestionPool AS QMS_QuestionPool_Parent
		ON QMS_AssessmentQuestion_Parent.QuestionID = QMS_QuestionPool_Parent.QuestionID
	LEFT OUTER JOIN APP_SampleSummary
		ON QMS_AssessmentQuestion.SampleSummaryID = APP_SampleSummary.SampleSummaryID
	LEFT OUTER JOIN
	(
		SELECT
			SelectedValue AS ParentCheckID,
			COUNT(SelectedID) As NoOfChildChecks
		FROM TEMP_Selected
		WHERE UserID = @UserID
			AND SelectedValue IS NOT NULL
		GROUP BY SelectedValue
	) AS NumberOfChildChecks
		ON TEMP_Selected.SelectedValue = NumberOfChildChecks.ParentCheckID
WHERE
	TEMP_Selected.UserID = @UserID
	AND QMS_ResponseQuestion.RecordStatus < 201
	AND QMS_AssessmentQuestion.HideFromReports = 0
	AND	EXISTS
	(
		SELECT ResponseID
		FROM #tmpShownAssessmentQuestions
		WHERE
			ResponseID = QMS_ResponseQuestion.ResponseID
			AND AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID
	)
ORDER BY ISNULL(TEMP_Selected.SelectedValue, QMS_ResponseQuestion.ResponseID), QMS_ResponseQuestion.ResponseID, AssessmentQuestion_Sort.SortIndex;

END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_Response_StockCodeID')
	EXEC('CREATE FUNCTION [dbo].[FNC_Response_StockCodeID]() RETURNS TABLE AS RETURN (SELECT 1 y)')
GO
-- =============================================
-- Author:		MarkP, MartinP, TimC
-- Create date: 25/03/2009
-- Description:	Gets stock code id for response
-- Version:		8
-- =============================================
-- JT TFS 7830 - Also link through raw stockcode datalookup
-- JE TFS 8684 - added filter so deleted Header items are ignored
-- MP TFS 12720: Added check on ResponseID to mitigate duplicates
-- JF TFS 15314 - timeout issues - changed duplicate check to an outer join (Greencell had 100k records in @ResponseHeader)
-- TC TFS 87211 - Re-written to remove the wrapper and improve performance
-- JF TFS 95146 - updated to make sure we pick up the stockcodeid rather than the Consignmentitemid (and remove duplicates)
--                COPIED FROM FNC_ResponseHeader_StockCodeID FOR HOTFIX
-- TC TFS 95787 - Create new function with parameters to avoid parameter sniffing when using external parameters
-- MartinP TFS 13-13907 Made into single use function for use with CROSS APPLY
-- JT TFS 95841 - Merged into FNC_Response_StockCodeID (identical functionality)
-- JF TFS 144121 -include final product 
-- =============================================
/*
    Example call:
    SELECT * FROM [dbo].[FNC_Response_StockCodeID](316)
*/
-- =============================================
ALTER FUNCTION [dbo].[FNC_Response_StockCodeID] 
(
	@ResponseID int
)
RETURNS TABLE
AS
RETURN 
SELECT	ResponseID, 
		StockCodeID
FROM (
		SELECT  R.ResponseID,
				ROW_NUMBER() OVER (
					PARTITION BY R.ResponseID 
					ORDER BY	CASE	WHEN AH.DataLookupID = 8	THEN 3		-- GoodsIntake (lowest precedence)
										WHEN AH.DataLookupID = 6	THEN 2		-- Consigment  
																	ELSE 1 END,	-- Stockcode or raw stockcode (highest precedence)
								AH.ListIndex,
								RH.ResponseHeaderID
				) AS row_order,
		CASE WHEN AH.DataLookupID in (6,8) THEN COALESCE(CI_Direct.StockCodeID, CI.StockCodeID) 
             ELSE RH.Header_SourceID END AS StockCodeID --make sure we only pick up the Header_SourceID when its a stockcodeid
      FROM QMS_Response R
      JOIN QMS_ResponseHeader RH ON R.ResponseID = RH.ResponseID
      JOIN QMS_AssessmentHeader AH ON RH.AssessmentHeaderID = AH.AssessmentHeaderID

          LEFT JOIN PM_VW_GoodsIntake GI ON RH.Header_SourceID = GI.GoodsIntakeID AND AH.DataLookupID = 8
                     LEFT OUTER JOIN PM_VW_ConsignmentItems CI ON GI.ConsignmentItemID = CI.ConsignmentItemID 
          LEFT JOIN PM_VW_ConsignmentItems CI_Direct ON CI_Direct.ConsignmentItemID = RH.Header_SourceID AND AH.DataLookupID = 6

     WHERE R.RecordStatus < 201
       AND RH.Header_SourceID IS NOT NULL
       AND AH.DataTypeID = 9
       AND AH.DataLookupID IN (2, 35, 6, 8,69) -- stockcode, raw stockcode, POitem (consignment item), intake, final product
       AND AH.RecordStatus < 201
       AND R.ResponseId = @ResponseId
	) AS tmp
WHERE tmp.row_order = 1
GO




PRINT 'Database updated successfully!'
GO