-- =============================================
-- Author:		Jeremy Formby
-- Create date: 3/05/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON

GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_User_ResponseList')
	EXEC('CREATE FUNCTION [dbo].[FNC_User_ResponseList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 17/08/2009
-- Description:	Returns list of responses that user can access
-- Version: 	19
-- =============================================
-- JT TFS 4194 - 14/08/2012 - When stock code is restricted, this now returns checks where no stock code is specified in the header.
-- JE TFS 4622 - returns QAFlagID based on ReportQAFlag config setting
-- JT TFS 5801 - performance improvements
-- JE TFS 5905 - updated to make use of Reporting Warehouse table
-- MP TFS 6031 - changed non-admin filter to ensure that temp checks are not included
-- MP TFS 6152 - fix to handle supplier site permissions
-- JT TFS 6689 - FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7612 - Void checks visibility is controlled by UserSetting for specific reports.
-- JE TFS 13-3570: added PRIMARY KEY to @Table_StockCode to improve performance
-- MP TFS 13-12783: fixed issue with ambiguous Location permission linking
-- jf tfs 16880 - UBE Check view performance for user with 2000+ stock code permissions - added lots of filters for updates like ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)
-- MP TFS 23409 - Added conditions in the CanUseWarehouse block so that Suppliers (userlevel=9) use supplier site code and non supplier (userlevel<>9) use locations
-- SS TFS 13-49452: Added logic so that when a client is using datawarehousing, if they are filtering on today's date, the warehouse is used for all dates up to today's date AND then today's date data
-- is worked out without the warehouse being used. This prevents timeouts FROM happening.
-- SS TFS 13-49897: Added ISNULL(Uploaded, EndDate) so that EndDate will be used if Uploaded contains a NULL Value
-- PH TSF 13-61027 : v16 18/11/2016 - Bug Fix.  When including checks from the datawarehouse for todays date, an assumption was made that the datawarehouse had been updated before @DateFrom.  
--                   However, if this was not the case then checks were included from the past (but after the warehouse had been updated).  The greater of the datawarehouse update time and @DateFrom is now used.
-- JT TFS 13-82627 : Fix issue of Uploaded results appearing on QC Checks screen not in the date range specified.
-- JT TFS 13-88313 : Only merge results from warehouse within the date range specified.
-- JF TFS 108475 : enddate is local time so calculate the offset before doing date comparrisons
-- =============================================
ALTER FUNCTION [dbo].[FNC_User_ResponseList]
(
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	ResponseStatus tinyint,
	QAFlagID int
)
AS
BEGIN

--declare @UserID int, @DateFrom datetime,	@DateTo datetime
--select @UserID = 836, @DateFrom = '1 Jun 2013 00:00:00',  @DateTo = '21 May 2014 23:59:59'

	DECLARE @UnfilteredResponse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	-- This will contain responses which are not in the warehouse, typically those with today's date.
	DECLARE @UnfilteredResponseNotInWarehouse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	--get date reporting warehouse table last built
	DECLARE @WarehouseBuildDate datetime
	SET @WarehouseBuildDate = (SELECT Updated FROM APP_Config WHERE OptionID = 78)

	DECLARE @UserSettingDisplayVoids bit 
	SET @UserSettingDisplayVoids = ISNULL(cast(dbo.FNC_GetUserSetting(@UserID, 'DisplayVoidChecks') as bit), 0)

	-- Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserSettingDisplayVoids = 1 -- Include all recordstatuses when we set a report to do the same by setting the User Setting.
		SET @RecordStatus = 255 -- maximum possible. 
	ELSE IF @UserLevel = 1  --Admin can always see hidden/void checks.
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 100

	-- Permissions that this user has.
	DECLARE @PermissionLink TABLE (
		UserID int, SupplierID int, CustomerID int, LocationID int, StockCodeID int, 
		StockCodeSubGroup nvarchar(100), BusinessGroup nvarchar(100)
	)
	INSERT INTO	@PermissionLink (UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup)
	SELECT	UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup
	FROM	VW_PermissionLink PermissionLink
	WHERE	PermissionLink.PermissionAllow = 1 AND PermissionLink.PermissionType IN (101,102,103,104,105,106) AND UserID = @UserID

	DECLARE @Table_StockCode TABLE (
		StockCodeID int PRIMARY KEY
	)
	
	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code, and translate to StockCodeID
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		BusinessGroup IN (SELECT BusinessGroup FROM @PermissionLink WHERE BusinessGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @PermissionLink WHERE StockCodeSubGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeID IN (SELECT StockCodeID FROM @PermissionLink WHERE StockCodeID IS NOT NULL)

	-- The datawarehouse can still be used when the user filters after the last warehouse build, but it will not contain much (if any) useful checks.
	-- We insert all the recent checks into @UnfilteredResponse so that the datawarehouse can still be used when the user filters on today's date.
	-- Just because the date range in entirely contained within the warehouse, doesn't mean that we only search in the warehouse. 
	-- Checks uploaded after the last warehouse build, but whose EndDate is within that range need to be included, as do incomplete checks (ResponseStatus 0)
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1
	BEGIN
		DECLARE @override bit 
		SET @override = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
		
		--get the min offset to catch all. Alternative method to calc each response UTC using UTCOffset would be too slow
		declare @MinOffset int 
		select @MinOffset = ISNULL(MIN(utcoffset),0) from qms_response where EndDate > DATEADD(MONTH,-3,GETDATE())

		DECLARE @ResponseFromDate datetime
		SELECT @ResponseFromDate = CASE WHEN @WarehouseBuildDate > @DateFrom THEN @WarehouseBuildDate ELSE @DateFrom END
		SELECT @ResponseFromDate = DATEADD(Hour,@MinOffset,@ResponseFromDate)
			
		-- Get the majority of checks which will not be in the warehouse - the ones added after the warehouse has been built.
		INSERT INTO @UnfilteredResponseNotInWarehouse (ResponseID, ResponseStatus, QAFlagID) 
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM	QMS_Response
		WHERE	(RecordStatus < @RecordStatus) 
		AND EndDate BETWEEN @ResponseFromDate AND @DateTo


		-- Make sure we include the checks uploaded after the warehouse was last built, but within the entire date range. They will not yet be in the warehouse.
		-- Also, we have to make sure that incomplete checks get included, since they are never put in the warehouse.
		INSERT INTO @UnfilteredResponseNotInWarehouse (ResponseID, ResponseStatus, QAFlagID) 
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM QMS_Response
		WHERE	(RecordStatus < @RecordStatus) 
		AND (Uploaded > @WarehouseBuildDate OR ResponseStatus = 0)
		AND EndDate BETWEEN @DateFrom AND @DateTo
		AND ResponseID NOT IN (SELECT ResponseID FROM @UnfilteredResponseNotInWarehouse)


		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponseTodaysDate
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
			ISNULL(UnfilteredResponseTodaysDate.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponseTodaysDate
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
			ISNULL(UnfilteredResponseTodaysDate.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponseTodaysDate
			SET		SupplierID = QMS_ResponseHeader.Header_SourceID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
			ISNULL(UnfilteredResponseTodaysDate.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponseTodaysDate
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponseTodaysDate.ResponseID = HeaderStockCode.ResponseID INNER JOIN
						PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponseTodaysDate.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponseTodaysDate
			SET		StockCodeID = HeaderStockCode.StockCodeID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponseTodaysDate.ResponseID = HeaderStockCode.ResponseID
			WHERE ISNULL(UnfilteredResponseTodaysDate.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)


		-- Include checks from the warehouse.
		INSERT INTO	@UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
		SELECT		RPT_Warehouse_Response.ResponseID, QMS_Response.ResponseStatus, RPT_Warehouse_Response.QAFlagID --Actual or Override QAFlag setting checked when building warehouse table
		FROM		RPT_Warehouse_Response 
		INNER JOIN  QMS_Response ON RPT_Warehouse_Response.ResponseID = QMS_Response.ResponseID
		WHERE		(QMS_Response.RecordStatus < @RecordStatus) 
		AND (RPT_Warehouse_Response.EndDate BETWEEN @DateFrom AND @DateTo)
		AND NOT EXISTS (SELECT UnfilteredResponseTodaysDate.ResponseID FROM @UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate
						WHERE UnfilteredResponseTodaysDate.ResponseID = RPT_Warehouse_Response.ResponseID)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponse
			SET	LocationID = RPT_Warehouse_Response.SupplierSiteID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.SupplierSiteID,0)
 
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponse
			SET	LocationID = RPT_Warehouse_Response.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponse
			SET		SupplierID = RPT_Warehouse_Response.SupplierID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(RPT_Warehouse_Response.SupplierID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponse
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID INNER JOIN
						PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponse
			SET		StockCodeID = RPT_Warehouse_Response.StockCodeID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID

		INSERT INTO @UnfilteredResponse(ResponseID, ResponseStatus, LocationID, CustomerID, SupplierID, StockCodeID, QAFlagID)
		SELECT ResponseID, ResponseStatus, LocationID, CustomerID, SupplierID, StockCodeID, QAFlagID FROM @UnfilteredResponseNotInWarehouse
	END
	ELSE --don't use warehouse table
	BEGIN
		DECLARE @OverrideOption bit 
		SET @OverrideOption = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
			
		INSERT INTO @UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @OverrideOption = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM	QMS_Response
		WHERE	(RecordStatus < @RecordStatus) AND (EndDate BETWEEN @DateFrom AND @DateTo)
			
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponse
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
			ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponse
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
			ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponse
			SET		SupplierID = QMS_ResponseHeader.Header_SourceID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
			ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponse
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID INNER JOIN
						PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponse
			SET		StockCodeID = HeaderStockCode.StockCodeID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID
			WHERE ISNULL(UnfilteredResponse.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)
	END

	-- Only return responses which match the filter specified, or for which we do not have a filter.
	-- This means that admin users (any user with no permissions set) can see all responses.
	INSERT INTO @Response (ResponseID, ResponseStatus, QAFlagID)
	SELECT	ResponseID, ResponseStatus, QAFlagID
	FROM	@UnfilteredResponse UnfilteredResponse
	WHERE (	NOT EXISTS (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL)
		OR	LocationID IN (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
		OR	SupplierID IN (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL)
		OR	CustomerID IN (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL) )
	AND (	StockCodeID IS NULL -- Allow unlinked stockcode responses
		OR	NOT EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		OR	StockCodeID IN (SELECT StockCodeID FROM @Table_StockCode) )
	
	RETURN
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter_CorrectiveAction')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter_CorrectiveAction] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 27/05/2014
-- Description:	Works out which ResponseIDs need to be put in the filter.
--              Can be used by any report.
-- Version:		3
-- =============================================
-- JT TFS 13-6955: Filtering can take place for a single tier (used to require both tiers for anything except location and supplier)
-- DPC TFS 13-104879: 05/09/2018 - Allow multiple choices in second level filter
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter_CorrectiveAction]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST1 nvarchar(4000),
	@FilterText_LIST1 nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID_LIST2 nvarchar(4000) = NULL,
	@FilterText_LIST2 nvarchar(4000) = NULL,
	
	@Offline bit = NULL,
	@SearchLocalChecksIfOffline bit = 1 -- Use QMS_Response even if offline (default)
AS
BEGIN 
	IF nullif(object_id('tempdb..#tmpUserCAs'), 0) IS NOT NULL DROP TABLE #tmpUserCAs
	CREATE TABLE #tmpUserCAs (
		CorrectiveActionID int,
		IsActive bit NULL
	)
	
	--temp table containing all CAs (i.e. all statuses) user has permission to see
	INSERT INTO	#tmpUserCAs (CorrectiveActionID, IsActive)
	SELECT	CorrectiveActionID, convert(bit, 1) as IsActive
	FROM	dbo.FNC_User_CorrectiveActionList(@UserID, 1)

	INSERT INTO	#tmpUserCAs (CorrectiveActionID, IsActive)
	SELECT	CorrectiveActionID, 0
	FROM	dbo.FNC_User_CorrectiveActionList(@UserID, 0)
	
	--strip out supplier and locations.
	DECLARE @SupplierFilter_LIST nvarchar(4000)
	SELECT @SupplierFilter_LIST = NULL
	IF (@ReportFilterID1 = 1) -- supplier
	BEGIN 
		SET @SupplierFilter_LIST = @FilterID_LIST1
		SELECT @ReportFilterID1 = NULL, @FilterID_LIST1 = NULL, @FilterText_LIST1 = NULL
	END
	IF (@ReportFilterID2 = 1) -- supplier
	BEGIN 
		SET @SupplierFilter_LIST =  @FilterID_LIST2
		SELECT @ReportFilterID2 = NULL, @FilterID_LIST2 = NULL, @FilterText_LIST2 = NULL
	END	
		
	DECLARE @LocationFilter_LIST nvarchar(4000)
	SELECT @LocationFilter_LIST = NULL
	IF (@ReportFilterID1 = 6) -- location
	BEGIN 
		SET @LocationFilter_LIST = @FilterID_LIST1
		SELECT @ReportFilterID1 = NULL, @FilterID_LIST1 = NULL, @FilterText_LIST1 = NULL
	END
	IF (@ReportFilterID2 = 6) -- location
	BEGIN 
		SET @LocationFilter_LIST = @FilterID_LIST2
		SELECT @ReportFilterID2 = NULL, @FilterID_LIST2 = NULL, @FilterID_LIST2 = NULL
	END	

	IF @SupplierFilter_LIST IS NOT NULL --filtering by Supplier
		DELETE #tmpUserCAs 
		WHERE CorrectiveActionID NOT IN (
			SELECT     CorrectiveActionID
			FROM         QMS_CorrectiveAction 
			LEFT OUTER JOIN	PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID 
			WHERE	(QMS_CorrectiveAction.RecordStatus < 201) 
			AND		(QMS_CorrectiveAction.AssignSupplierID IN (SELECT Token FROM dbo.FNC_SplitString(@SupplierFilter_LIST, ','))) 
			AND		(QMS_CorrectiveAction.DateOccurred >= @DateFrom)
		)
		
	IF @LocationFilter_LIST IS NOT NULL --filtering by Supplier
		DELETE #tmpUserCAs 
		WHERE CorrectiveActionID NOT IN (
			SELECT     CorrectiveActionID
			FROM         QMS_CorrectiveAction 
			LEFT OUTER JOIN	PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID				  
			WHERE	(QMS_CorrectiveAction.RecordStatus < 201) 
			AND		(QMS_CorrectiveAction.LocationID IN (SELECT Token FROM dbo.FNC_SplitString(@LocationFilter_LIST, ',')))
			AND		(QMS_CorrectiveAction.DateOccurred >= @DateFrom)
		)
	
	IF @ReportFilterID1 IS NOT NULL OR @ReportFilterID2 IS NOT NULL  --not filtering by Supplier
	BEGIN
		-- To get response filters
		EXEC APP_Temp_ReportFilter @UserID,	@DateFrom, @DateTo, @AssessmentTypeID_LIST, 
									@ReportFilterID1, @FilterID_LIST1, @FilterText_LIST1,
									@ReportFilterID2, @FilterID_LIST2, @FilterID_LIST2,
									@Offline, @SearchLocalChecksIfOffline

		DELETE #tmpUserCAs 
		WHERE CorrectiveActionID NOT IN (
			SELECT UserCAList.CorrectiveActionID
			FROM #tmpUserCAs UserCAList
			INNER JOIN dbo.FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID
			INNER JOIN TEMP_ResponseFilter ON CAResponseList.ResponseID = TEMP_ResponseFilter.ResponseID AND UserID = @UserID
		)
	END


	DELETE TEMP_CorrectiveActionFilter WHERE UserID = @UserID
	INSERT TEMP_CorrectiveActionFilter (CorrectiveActionID, UserID)
	SELECT QMS_CorrectiveAction.CorrectiveActionID, @UserID FROM #tmpUserCAs
	INNER JOIN QMS_CorrectiveAction ON QMS_CorrectiveAction.CorrectiveActionID = #tmpUserCAs.CorrectiveActionID
	WHERE (QMS_CorrectiveAction.RecordStatus < 201)
	AND (@DateFrom IS NULL OR DateOccurred > @DateFrom) --OR DateDeadline > @DateFrom OR DateCompleted > @DateFrom OR DateFailed > @DateFrom)
	AND (@DateTo IS NULL OR DateOccurred <= @DateTo)-- OR DateDeadline <= @DateTo OR DateCompleted <= @DateTo OR DateFailed <= @DateTo)
	
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter_StockCode')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter_StockCode] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which Stockcodes need to be put in the filter.
--              Can be used by any report.
-- Version:		2
-- =============================================
-- JT TFS 13-40008: 22/02/2016 - Extracted from APP_Temp_ReportFilter for speed reasons.
-- DPC TFS 13-104879: 05/09/2018 - Allow multiple choices in second level filter
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter_StockCode]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST1 nvarchar(4000),
	@FilterText_LIST1 nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID_LIST2 nvarchar(4000) = NULL,
	@FilterText_LIST2 nvarchar(4000) = NULL,
	
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	
	DELETE TEMP_StockCodeFilter WHERE UserID = @UserID

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(ISNULL(@FilterID_LIST1, '')) = 0 AND
		LEN(ISNULL(@FilterText_LIST1, '')) = 0 AND
		@ReportFilterID2 IS NOT NULL AND
		(LEN(ISNULL(@FilterID_LIST2, '')) > 0 OR LEN(ISNULL(@FilterText_LIST2, '')) > 0)
	)
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST1 = @FilterID_LIST2,
				@FilterText_LIST1 = @FilterText_LIST2
				
		SELECT @ReportFilterID2 = NULL, @FilterID_LIST2 = NULL, @FilterText_LIST2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF (@ReportFilterID2 IS NOT NULL) AND (LEN(ISNULL(@FilterID_LIST2, '')) = 0) AND (LEN(ISNULL(@FilterText_LIST2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))
	INSERT @FilterItem (pos, id, txt)
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST1, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST1, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 
	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END

	DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)

	-- Build up a table of stockcodes using the first level filter
	DECLARE @FilteredStockCode TABLE (id int not null)
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN		
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		INSERT @FilteredStockCode (id)
		SELECT StockCodeID
		FROM dbo.FNC_ReportFilter_StockCode(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo)
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN 
	
		-- Build up a table of stockcodes using the second level filter		
		DECLARE @FilterItem2 TABLE (pos int not null, id int, txt nvarchar(255))
		INSERT @FilterItem2 (pos, id, txt)	
		SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
		FROM dbo.FNC_SplitString(@FilterID_LIST2, ',') a
		FULL JOIN dbo.FNC_SplitString(@FilterText_LIST2, ',') b on a.Position = b.Position

		DECLARE @FilteredStockCode2 TABLE (id int not null)
		WHILE EXISTS (SELECT * FROM @FilterItem2)
		BEGIN
			SELECT @pos = pos,  @FilterID = id, @FilterText = txt
			FROM @FilterItem2
			WHERE pos = (SELECT MIN(pos) FROM @FilterItem2)

			INSERT @FilteredStockCode2 (id)
			SELECT StockCodeID
			FROM dbo.FNC_ReportFilter_StockCode(@ReportFilterID2, @FilterID, @FilterText, @DateFrom, @DateTo)
		
			DELETE @FilterItem2 WHERE pos = @pos
		END
		
		-- remove stockcodes from the first filter set that are not in the second filter set
		DELETE @FilteredStockCode 
		WHERE id NOT IN (SELECT id FROM @FilteredStockCode2)

	END

	-- Only return stock codes that the user is allowed to see
	INSERT TEMP_StockCodeFilter (UserID, StockCodeID)
	SELECT @UserID as UserID, fs.id
	FROM @FilteredStockCode fs 
	INNER JOIN FNC_User_StockCodeList(@UserID) sl ON fs.id = sl.StockCodeID 
	--GROUP BY fs.id --removed in TFS 4740 as this was slowing the query down a LOT on JS UAT
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter_StockCode '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST1 = ' + CASE WHEN LEN(@FilterID_LIST1)=0 THEN 'NULL' ELSE @FilterID_LIST1 END
			SET @ReportString = @ReportString + ', @FilterText_LIST1 = ' + CASE WHEN LEN(@FilterText_LIST1)=0 THEN 'NULL' ELSE @FilterText_LIST1 END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST2 = ' + CASE WHEN LEN(@FilterID_LIST2)=0 THEN 'NULL' ELSE @FilterID_LIST2 END
			SET @ReportString = @ReportString + ', @FilterText_LIST2 = ' + CASE WHEN LEN(@FilterText_LIST2)=0 THEN 'NULL' ELSE @FilterText_LIST2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @UserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which ResponseIDs need to be put in the filter.
--              Can be used by any report.
-- Version:		25
-- =============================================
-- j.toolan 30-mar-2012: If @AssessmentTypeID_LIST is empty, then join all assessments
-- j.toolan 13 Apr 2012: Fixed bug with 2nd filter incorrectly clearing.
-- j.toolan 18 Apr 2012: Added Code to filter by stockcode where appropriate (renamed to ReportFilter)
-- JE 25 Apr 2012: made ReportFilterID1 optional, so can just filter by AssessmentType if necessary (TFS 3382)
-- j.toolan 6 July 2012: TFS 4013: Dynamic filtering added.
-- JE TFS 4740: removed GROUP BY for performance reasons
-- JE TFS 5037: corrected @ReportFilterID2 check as 2nd tier filter wasn't working
-- MP TFS 5289: changed date format used for custom filter functions
-- JT TFS 5396: changed code to allow commas in FilterTexts passed in as CSV. (replace "," with "&comma;" )
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JT TFS 8117: Added offline filtering.
-- MP TFS 8227: Made temp table user specific and added performance logging
-- JE TFS 8274: Put result set of FNC_User_ResponseList into temp table before using it
-- JT TFS 13-667: Do not filter responses if the date range is too big.
-- JT TFS 13-1076: Search either QMS_Response or QMS_SupplierResponse, but not both.
-- JF TFS 2281 - if filter doesn't have a sourceid, this wasn't working
-- JT TFS 13-17467: Only include checks which are eligible candidates for the data warehouse.
-- JT TFS 13-38578: Convert SP to use QRY_User_ResponseList (rather than FNC_UserResponseList)
-- JT TFS 13-40008: 22/02/2016 - Added transaction IDs to response list generation, and extracted stockcode filtering
-- JT TFS 13-42222: 02/03/2016 - Added link to TokenUserID so that reports can be run simultaneously.
-- JF TFS 35438 - make sure we remove data which wouldn't show in the data warehouse - ie parent checks for Assessment type 5
-- jf TFS 90761 - Morrisons QC Office - Issue with reports not generating - removed the use of FNC_Response_ShouldIncludeInDataWarehouse dues to performance issues 
-- CJP TFS 87358 - Appending FilterText to TEMP_UserData table to ensure it is visible on report
-- jf TFS 102053 - increased GroupByText nvarchar(50) to GroupByText nvarchar(255)
-- DPC TFS 13-104879: 05/09/2018 - Allow multiple choices in second level filter
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST1 nvarchar(4000),
	@FilterText_LIST1 nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID_LIST2 nvarchar(4000) = NULL,
	@FilterText_LIST2 nvarchar(4000) = NULL,
	
	@Offline bit = NULL,
	@SearchLocalChecksIfOffline bit = 1, -- Use QMS_Response even if offline (default)
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	--//This can be removed when fixed properly
	--Insert the filter text into TEMP_UserData for retreival later
	DELETE FROM TEMP_UserData where @UserID=@UserID
	INSERT INTO TEMP_UserData (UserID, String1) VALUES (@UserID, CASE WHEN LEN(@FilterText_LIST1)>100 THEN LEFT(@FilterText_LIST1,97)+ '...' ELSE LEFT(@FilterText_LIST1,100) END)
	--//End of fix
	
	EXEC SAVE_TEMP_Transaction @TransactionID, @UserID

	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	IF @Offline IS NULL SELECT @Offline = CASE WHEN db_name() = 'GreenlightQC_Mobile' THEN 1 ELSE 0 END
	
	DECLARE @SearchOffline bit -- if this is set, then we search in QMS_SupplierResponse.
	SELECT @SearchOffline  = CASE WHEN @Offline = 1 AND @SearchLocalChecksIfOffline = 0 THEN 1 ELSE 0 END
		
	-- Do not filter responses if the date range is not specified
	DECLARE @performResponseFiltering bit 
	SELECT @performResponseFiltering = CASE WHEN @DateFrom <= '1 Jan 2001' and datediff(year, @DateFrom, @DateTo) >= 10 THEN 0 ELSE 1 END
	
	DELETE TEMP_ResponseFilter WHERE UserID = @TokenUserID OR TransactionID = @TransactionID

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(ISNULL(@FilterID_LIST1, '')) = 0 AND
		LEN(ISNULL(@FilterText_LIST1, '')) = 0 AND
		@ReportFilterID2 IS NOT NULL AND
		(LEN(ISNULL(@FilterID_LIST2, '')) > 0 OR LEN(ISNULL(@FilterText_LIST2, '')) > 0)
	)
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST1 = @FilterID_LIST2,
				@FilterText_LIST1 = @FilterText_LIST2
				
		SELECT @ReportFilterID2 = NULL, @FilterID_LIST2 = NULL, @FilterText_LIST2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF (@ReportFilterID2 IS NOT NULL) AND (LEN(ISNULL(@FilterID_LIST2, '')) = 0) AND (LEN(ISNULL(@FilterText_LIST2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))
	INSERT @FilterItem (pos, id, txt)
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST1, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST1, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 
	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END
	
	DECLARE @customSql nvarchar(1000)
	SET @customSql = NULL
	CREATE TABLE #tmpCustomFilter (UserID INT, ResponseID int, GroupByText nvarchar(255))	
	DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)		
	DECLARE @CustomFilterFunction nvarchar(50)
	SET @CustomFilterFunction = NULL

	-- Do some custom filtering if it is set up. This acts like "AND" (restricting results)
	SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
	FROM APP_ReportFilter 
	WHERE ReportFilterID = @ReportFilterID1

	-- Build up a table of responses using the first level filter
	DECLARE @FilteredResponse TABLE (id int not null)	
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN		
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		IF @performResponseFiltering = 1
		BEGIN
			-- either the custom function runs, *OR* the generic one, *OR* the offline one, but not more than one.
			IF @SearchOffline = 1
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				SELECT @customSql =
					'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText)
					SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText
					FROM dbo.[' + @CustomFilterFunction + ']' + 
										'(' + convert(nvarchar(10), @UserID) + ', ' +
										convert(nvarchar(30), @FilterID) + ', ' +
										'''' + convert(nvarchar(50), @FilterText) + ''', ' + 
										'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
										'''' + convert(nvarchar(30), @DateTo, 13) + ''');'				
				IF @customSql IS NOT NULL EXECUTE (@customSql)
			END
			ELSE
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
		END
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- insert all our new custom filtered items (added to temp table in loop above)
	INSERT @FilteredResponse (id)
	SELECT ResponseID  FROM #tmpCustomFilter WHERE UserID = @TokenUserID

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN
	
		-- Build up a table of responses using the second level filter
		DELETE #tmpCustomFilter WHERE UserID = @TokenUserID --clear it out for second filter.	
		DECLARE @FilterItem2 TABLE (pos int not null, id int, txt nvarchar(255))
		INSERT @FilterItem2 (pos, id, txt)	
		SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
		FROM dbo.FNC_SplitString(@FilterID_LIST2, ',') a
		FULL JOIN dbo.FNC_SplitString(@FilterText_LIST2, ',') b on a.Position = b.Position
		
		-- Do some custom filtering if it is set up. This acts like "AND" (restricting results)
		SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
		FROM APP_ReportFilter 
		WHERE ReportFilterID = @ReportFilterID2		
		
		DECLARE @FilteredResponse2 TABLE (id int)
		
		WHILE EXISTS (SELECT * FROM @FilterItem2)
		BEGIN	
			SELECT @pos = pos,  @FilterID = id, @FilterText = txt
			FROM @FilterItem2
			WHERE pos = (SELECT MIN(pos) FROM @FilterItem2)
			
			IF @performResponseFiltering = 1
			BEGIN
				-- either the custom function runs, *OR* the generic one, *OR* the offline one, but not more than one.						
				IF @SearchOffline = 1
				BEGIN
					INSERT @FilteredResponse2 (id)
					SELECT ResponseID 
					FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID2, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
				END 
				ELSE IF @CustomFilterFunction IS NOT NULL
				BEGIN									
					SELECT @customSql =
						'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText)
						SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText
						FROM dbo.[' + @CustomFilterFunction + ']' + 
											'(' + convert(nvarchar(10), @UserID) + ', ' +
											convert(nvarchar(30), @FilterID) + ', ' +
											'''' + convert(nvarchar(50), @FilterText) + ''', ' + 
											'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
											'''' + convert(nvarchar(30), @DateTo, 13) + ''');'
					IF @customSql IS NOT NULL EXECUTE (@customSql)					
				END
				ELSE
				BEGIN
					INSERT @FilteredResponse2 (id)
					SELECT ResponseID 
					FROM dbo.FNC_ReportFilter_Response(@ReportFilterID2, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
				END			
			END
			
			DELETE @FilterItem2 WHERE pos = @pos
		END
		
		-- insert all our new custom filtered items (added to temp table in loop above)
		INSERT @FilteredResponse2 (id)
		SELECT ResponseID FROM #tmpCustomFilter WHERE UserID = @TokenUserID
		
		-- remove responses from the first filter set that are not in the second filter set
		DELETE @FilteredResponse
		WHERE id NOT IN (SELECT id FROM @FilteredResponse2)

	END

	IF @performResponseFiltering = 1
	BEGIN
		IF @SearchOffline = 1
		BEGIN
			IF @ReportFilterID1 IS NOT NULL
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY r.ResponseID
			ELSE
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id	
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY r.ResponseID
		END
		ELSE
		BEGIN
			--JE TFS 82774 - temp table of responses user can view (moved from direct access in query to improve performance)
			IF nullif(object_id('tempdb..#tmpUserResponses'), 0) IS NOT NULL DROP TABLE #tmpUserResponses
			CREATE TABLE #tmpUserResponses (UserID int, ResponseID int, ResponseStatus int, QAFlagID int)
			INSERT #tmpUserResponses (UserID, ResponseID, ResponseStatus, QAFlagID)
			EXEC QRY_User_ResponseList @UserID, @DateFrom, @DateTo
			
			IF @ReportFilterID1 IS NOT NULL
				-- Only return responses that the user is allowed to see, filter by Assessment Type, and use filter
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE rl.UserID = @UserID
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY rl.ResponseID
			
			IF @ReportFilterID1 IS NULL
				-- Only return responses that the user is allowed to see, but there has been no filter specified, so only filter by Assessment Type (if specified)
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				WHERE rl.UserID = @UserID	
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY rl.ResponseID
			
			DROP TABLE #tmpUserResponses
		END

	END
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST1 = ' + CASE WHEN LEN(@FilterID_LIST1)=0 THEN 'NULL' ELSE @FilterID_LIST1 END
			SET @ReportString = @ReportString + ', @FilterText_LIST1 = ' + CASE WHEN LEN(@FilterText_LIST1)=0 THEN 'NULL' ELSE @FilterText_LIST1 END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST2 = ' + CASE WHEN LEN(@FilterID_LIST2)=0 THEN 'NULL' ELSE @FilterID_LIST2 END
			SET @ReportString = @ReportString + ', @FilterText_LIST2 = ' + CASE WHEN LEN(@FilterText_LIST2)=0 THEN 'NULL' ELSE @FilterText_LIST2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_TEMP_ReportGroupBy_Response')
	EXEC ('CREATE PROCEDURE [dbo].[APP_TEMP_ReportGroupBy_Response] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out the ResponseIDs need to be put in the filter. Created for GroupBy Custom Functions
--              Can be used by any report.
-- Version:		9
-- =============================================
-- JT TFS 5358 - Created
-- JE TFS 5363 - updated to include GroupByHeader
-- JT TFS 5546 - Updated to include custom GroupByHeaders
-- JT TFS 6139 - Increased size of individual result, to account for extra long stockcode descriptions.
-- JT TFS 8117 - Added Offline grouping
-- MP TFS 8227 - made temp tables user specific
-- JT TFS 13-1076: Search either QMS_Response or QMS_SupplierResponse, but not both.
-- JT TFS 13-38578: Only update #tmpGroupBy temp table if there's a change to the data.
-- DPC TFS 13-104879: Fixed call to APP_Temp_ReportFilter to include missing NULL second level filter parameters
-- =============================================
ALTER PROCEDURE [dbo].[APP_TEMP_ReportGroupBy_Response]
	@UserID int,
	@DateFrom datetime,							-- Required for GroupBy functions.
	@DateTo datetime,							-- Required for GroupBy functions.
	@GroupByFilterList nvarchar(255),			-- ReportFilterIDs that we want to group by, in order.
	@IsFilterApplied bit = 1,					-- Do we use the values from previous filter, or generate our own list from the extra parameters below:
	@AssessmentTypeID_LIST nvarchar(255) = NULL,--(only used if @IsFilterApplied = 0)
	@Offline bit = NULL,						-- Do we include offline (QMS_SupplierResponse) checks?
	@SearchLocalChecksIfOffline bit = 1			-- Use QMS_Response even if offline (default)
AS
BEGIN
	SET NOCOUNT ON

	-- If they didn't specify the offline flag, then try to work it out.
	IF @Offline IS NULL SELECT @Offline = CASE WHEN db_name() = 'GreenlightQC_Mobile' THEN 1 ELSE 0 END
	
	DECLARE @SearchOffline bit -- if this is set, then we search in QMS_SupplierResponse.
	SELECT @SearchOffline  = CASE WHEN @Offline = 1 AND @SearchLocalChecksIfOffline = 0 THEN 1 ELSE 0 END
	
	-- Table to hold the final groupby information which will be built up along the way.
	IF NULLIF(object_id('tempdb..#tmpGroupBy'), 0) IS NOT NULL DROP TABLE #tmpGroupBy
	CREATE TABLE #tmpGroupBy (
		UserID INT,
		ResponseID int,
		GroupByText nvarchar(3750), -- note: dates and integers should be converted into a sortable format.
		GroupByHeader nvarchar(100)
	)


	-- Table to hold the function names which we are going to call to populate the group by information, and the order we apply them in.
	DECLARE @GroupByFunction TABLE (
		GroupByID int,
		GroupByFunction nvarchar(100),
		GroupByHeader nvarchar(100),
		ListIndex int
	)
	INSERT @GroupByFunction (GroupByID, GroupByFunction, GroupByHeader, ListIndex)
	SELECT	convert(int, splitString.Token) as GroupByID, 
			APP_ReportFilter.FilterFunction,
			ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), ReportFilter) AS GroupByHeader,
			splitString.Position 
	FROM dbo.FNC_SplitString(@GroupByFilterList, ',') as splitString
	LEFT OUTER JOIN APP_ReportFilter ON APP_ReportFilter.ReportFilterID = convert(int, splitString.Token)
    WHERE isnumeric(splitString.Token) = 1 
    ORDER BY splitString.Position 
	
	-- Which responses do we need to filter on, anyway?
	IF (@IsFilterApplied = 1)
	BEGIN
		INSERT #tmpGroupBy (UserID, ResponseID) SELECT @UserID AS UserID, ResponseID FROM TEMP_ResponseFilter WHERE UserID = @UserID GROUP BY ResponseID
	END
	ELSE
	BEGIN
		-- Run a generic filter ourselves (the front end should perhaps do this)
		DECLARE @TempUserID int 
		SET @TempUserID = @UserID
		EXEC APP_Temp_ReportFilter @TempUserID, @DateFrom, @DateTo, @AssessmentTypeID_LIST, NULL, NULL, NULL, NULL, NULL, NULL, @Offline, @SearchLocalChecksIfOffline
		INSERT #tmpGroupBy (UserID, ResponseID) SELECT @UserID AS UserID, ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TempUserID
	END
	
	IF NULLIF(object_id('tempdb..#tmpSingleGroupResult'), 0) IS NOT NULL DROP TABLE #tmpSingleGroupResult
	-- Do some custom grouping if it is set up. This acts like comma seperating results. i.e. Group by BusinessGroup, Department
	CREATE TABLE #tmpSingleGroupResult (
		UserID INT,
		ResponseID int,
		GroupByText nvarchar(255)
	)
	-- remember temp_selected, and set it back to how it was when we started.
	IF NULLIF(object_id('tempdb..#tmpOldSelected'), 0) IS NOT NULL DROP TABLE #tmpOldSelected
	SELECT UserID, SelectedID, SelectedText, SelectedValue INTO #tmpOldSelected FROM TEMP_Selected WHERE UserID = @UserID
	DELETE TEMP_Selected WHERE UserID = @UserID
	INSERT TEMP_Selected (SelectedID, UserID) 
	SELECT ResponseID, @UserID FROM #tmpGroupBy WHERE UserID = @UserID GROUP BY ResponseID
	
	WHILE EXISTS (SELECT * FROM @GroupByFunction)
	BEGIN
		DECLARE @index int, @groupFunction nvarchar(100), @groupByID int, @groupByHeader nvarchar(100), @customSql nvarchar(1000)
		SELECT	@index = min(ListIndex)
		FROM @GroupByFunction
		
		SELECT 	@groupByID = GroupByID,
				@groupByHeader = GroupByHeader,
				@groupFunction = GroupByFunction,
				@customSql = NULL
		FROM @GroupByFunction
		WHERE ListIndex = @index
	
		-- either the custom function runs, *OR* the generic, *OR* the offline, but not more than one.
		IF @SearchOffline = 1
		BEGIN
			INSERT #tmpSingleGroupResult (UserID, ResponseID, GroupByText)
			SELECT @UserID AS UserID, ResponseID, GroupByText FROM dbo.FNC_ReportGrouping_Response_Offline(@UserID, @groupByID)
			WHERE ResponseID NOT IN (SELECT ResponseID FROM #tmpSingleGroupResult WHERE UserID = @UserID)
		END 
		ELSE IF @groupFunction IS NOT NULL
		BEGIN
			SELECT @customSql = 'INSERT #tmpSingleGroupResult (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @UserID) + ', ResponseID, GroupByText FROM dbo.[' + @groupFunction + ']' + 
								'(' + convert(nvarchar(10), @UserID) + ', NULL, NULL, NULL, NULL);' 
			
			IF @customSql IS NOT NULL
				EXECUTE (@customSql)
		END
		ELSE
		BEGIN
			INSERT #tmpSingleGroupResult (UserID, ResponseID, GroupByText)
			SELECT @UserID AS UserID, ResponseID, GroupByText FROM dbo.FNC_ReportGrouping_Response(@UserID, @groupByID)
		END
		
		UPDATE tmp
		SET tmp.GroupByText = CASE WHEN tmp.GroupByText IS NULL THEN '' ELSE tmp.GroupByText + '|' END + ISNULL(single.GroupByText, ''),
			tmp.GroupByHeader = CASE WHEN tmp.GroupByHeader IS NULL THEN '' ELSE tmp.GroupByHeader + '|' END + ISNULL(@GroupByHeader, '')
		FROM #tmpGroupBy tmp
		INNER JOIN #tmpSingleGroupResult single ON tmp.ResponseID = single.ResponseID AND tmp.UserID = single.UserID

		INSERT #tmpGroupBy (UserID, ResponseID, GroupByText, GroupByHeader)
		SELECT @UserID AS UserID, single.ResponseID, ISNULL(single.GroupByText, ''), ISNULL(@GroupByHeader, '')
		FROM #tmpSingleGroupResult single
		WHERE NOT EXISTS (SELECT ResponseID FROM #tmpGroupBy WHERE ResponseID = single.ResponseID AND UserID = single.UserID)
	
		DELETE #tmpSingleGroupResult -- Yes, the whole table.
		DELETE @GroupByFunction WHERE ListIndex = @index
	END	
	
	-- Set TEMP_selected back to how it was:
	DELETE TEMP_Selected WHERE UserID = @UserID
	INSERT TEMP_Selected (UserID, SelectedID, SelectedText, SelectedValue) SELECT UserID, SelectedID, SelectedText, SelectedValue FROM #tmpOldSelected
	IF NULLIF(object_id('tempdb..#tmpOldSelected'), 0) IS NOT NULL DROP TABLE #tmpOldSelected

	-- Finally, update the groupby table in a swift pincer-like motion (quickly, that is).
	DELETE TEMP_ResponseGroupBy WHERE UserID = @UserID
	
	SET NOCOUNT OFF
	INSERT TEMP_ResponseGroupBy (UserID, ResponseID, GroupByHeader, GroupByText)
	SELECT @UserID, ResponseID, GroupByHeader, GroupByText FROM #tmpGroupBy WHERE UserID = @UserID
END
GO




PRINT 'Database updated successfully!'
GO

