-- =============================================
-- Author:		Jeremy Formby
-- Create date: 3/05/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON

GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO



-- BEGIN 01_FNC_Response_UserList_Indexes

--Update of existing index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_QMS_ResponseHeader_ResponseID') DROP INDEX [IX_QMS_ResponseHeader_ResponseID] ON [dbo].[QMS_ResponseHeader]
CREATE NONCLUSTERED INDEX [IX_QMS_ResponseHeader_ResponseID] ON [dbo].[QMS_ResponseHeader]([ResponseID],[Header_SourceID],[RecordStatus]) INCLUDE(AssessmentHeaderID)

--New index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_QMS_User_LeftCompanyRecordStatus') DROP INDEX [IX_QMS_User_LeftCompanyRecordStatus] ON [dbo].[QMS_User]
CREATE NONCLUSTERED INDEX [IX_QMS_User_LeftCompanyRecordStatus] ON [dbo].[QMS_User] ([LeftCompany],[RecordStatus]) INCLUDE ([UserID])

--Update of existing index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_QMS_UserPermissions_PermissionType') DROP INDEX [IX_QMS_UserPermissions_PermissionType] ON [dbo].[QMS_UserPermissions]
CREATE NONCLUSTERED INDEX [IX_QMS_UserPermissions_PermissionType] ON [dbo].[QMS_UserPermissions]([PermissionType],[PermissionAllow]) INCLUDE ([UserPermissionID],[UserID],[PermissionTitle],[Updated],[GUID],[RowVersion],[Created])

--New index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_PM_StockCode_RecordStatus') DROP INDEX [IX_PM_StockCode_RecordStatus] ON [dbo].[PM_StockCode] 
CREATE NONCLUSTERED INDEX [IX_PM_StockCode_RecordStatus] ON [dbo].[PM_StockCode] ([RecordStatus]) INCLUDE ([StockCodeID],[CustomerID])

--New index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_PM_StockCode_BusinessGroupRecordStatus') DROP INDEX [IX_PM_StockCode_BusinessGroupRecordStatus] ON [dbo].[PM_StockCode] 
CREATE NONCLUSTERED INDEX [IX_PM_StockCode_BusinessGroupRecordStatus] ON [dbo].[PM_StockCode] ([BusinessGroup],[RecordStatus],[StockCodeID])

--New index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_PM_StockCode_StockCodeSubGroupRecordStatus') DROP INDEX [IX_PM_StockCode_StockCodeSubGroupRecordStatus] ON [dbo].[PM_StockCode] 
CREATE NONCLUSTERED INDEX [IX_PM_StockCode_StockCodeSubGroupRecordStatus] ON [dbo].[PM_StockCode] ([StockCodeSubGroup],[RecordStatus],[StockCodeID])

--New index
IF EXISTS (SELECT * FROM sys.indexes WHERE name = N'IX_PM_StockCode_StockCodeSubGroupRecordStatus') DROP INDEX [IX_PM_StockCode_StockCodeSubGroupRecordStatus] ON [dbo].[PM_StockCode] 
CREATE NONCLUSTERED INDEX [IX_PM_StockCode_StockCodeSubGroupRecordStatus] ON [dbo].[PM_StockCode] ([CustomerId],[RecordStatus],[StockCodeID]) 

GO
-- END 01_FNC_Response_UserList_Indexes

-- BEGIN 02_FNC_ResponseHeader_CustomerID
IF NULLIF(OBJECT_ID('dbo.FNC_ResponseHeader_CustomerID'), 0) IS NOT NULL DROP FUNCTION [dbo].[FNC_ResponseHeader_CustomerID] 
GO
-- =============================================
-- Author:		Tim Cooke
-- Create date: 23/01/2017
-- Description:	Gets all supplier sites that should be notified for a check
-- Version:		1
-- =============================================
/*
    Example call:
    SELECT * FROM [dbo].[FNC_TIM_PRT_ResponseHeader_CustomerID]()
*/
-- =============================================
CREATE FUNCTION [dbo].[FNC_ResponseHeader_CustomerID] ()
RETURNS TABLE
AS
RETURN

    SELECT R.ResponseID
         , RH.Header_SourceID AS CustomerID
      FROM QMS_Response R
      JOIN QMS_ResponseHeader RH ON R.ResponseID = RH.ResponseID
      JOIN QMS_AssessmentHeader AH ON RH.AssessmentHeaderID = AH.AssessmentHeaderID
     WHERE R.RecordStatus < 201
       AND RH.Header_SourceID IS NOT NULL
       AND AH.DataTypeID = 9
       AND AH.DataLookupID = 3
       AND AH.RecordStatus < 201

GO
-- END 02_FNC_ResponseHeader_CustomerID

-- BEGIN 03_FNC_ResponseHeader_StockCodeID

IF NULLIF(OBJECT_ID('dbo.FNC_ResponseHeader_StockCodeID'), 0) IS NOT NULL DROP FUNCTION [dbo].[FNC_ResponseHeader_StockCodeID];
GO
-- =============================================
-- Author:		Markp
-- Create date: 25/03/2009
-- Description:	Gets stock code id for responses
-- Version:		7
-- =============================================
-- JT TFS 7830 - Also link through raw stockcode datalookup
-- JE TFS 8684 - added filter so deleted Header items are ignored
-- MP TFS 12720: Added check on ResponseID to mitigate duplicates
-- JF TFS 15314 - timeout issues - changed duplicate check to an outer join (Greencell had 100k records in @ResponseHeader)
-- TC TFS 87211 - Re-written to remove the wrapper and improve performance
-- JF TFS 95146 - updated to make sure we pick up the stockcodeid rather than the Consignmentitemid (and remove duplicates)
-- =============================================
/*
    Example call:
    SELECT * FROM [dbo].[FNC_ResponseHeader_StockCodeID]()
*/
-- =============================================
CREATE FUNCTION [dbo].[FNC_ResponseHeader_StockCodeID]()
RETURNS TABLE
AS
RETURN 
SELECT	ResponseID, 
		StockCodeID
FROM (
		SELECT  R.ResponseID,
				ROW_NUMBER() OVER (
					PARTITION BY R.ResponseID 
					ORDER BY	CASE	WHEN AH.DataLookupID = 8	THEN 3		-- GoodsIntake (lowest precedence)
										WHEN AH.DataLookupID = 6	THEN 2		-- Consigment  
																	ELSE 1 END,	-- Stockcode or raw stockcode (highest precedence)
								AH.ListIndex,
								RH.ResponseHeaderID
				) AS row_order,
		CASE WHEN AH.DataLookupID in (6,8) THEN COALESCE(CI_Direct.StockCodeID, CI.StockCodeID) 
             ELSE RH.Header_SourceID END AS StockCodeID --make sure we only pick up the Header_SourceID when its a stockcodeid
      FROM QMS_Response R
      JOIN QMS_ResponseHeader RH ON R.ResponseID = RH.ResponseID
      JOIN QMS_AssessmentHeader AH ON RH.AssessmentHeaderID = AH.AssessmentHeaderID

          LEFT JOIN PM_VW_GoodsIntake GI ON RH.Header_SourceID = GI.GoodsIntakeID AND AH.DataLookupID = 8
                     LEFT OUTER JOIN PM_VW_ConsignmentItems CI ON GI.ConsignmentItemID = CI.ConsignmentItemID 
          LEFT JOIN PM_VW_ConsignmentItems CI_Direct ON CI_Direct.ConsignmentItemID = RH.Header_SourceID AND AH.DataLookupID = 6

     WHERE R.RecordStatus < 201
       AND RH.Header_SourceID IS NOT NULL
       AND AH.DataTypeID = 9
       AND AH.DataLookupID IN (2, 35, 6, 8) -- stockcode, raw stockcode, POitem (consignment item), intake
       AND AH.RecordStatus < 201

	) AS tmp
WHERE tmp.row_order = 1
GO
-- END 03_FNC_ResponseHeader_StockCodeID

-- BEGIN 04_FNC_ResponseHeader_SupplierSiteID
IF NULLIF(OBJECT_ID('dbo.FNC_ResponseHeader_SupplierSiteID'), 0) IS NOT NULL DROP FUNCTION [dbo].[FNC_ResponseHeader_SupplierSiteID] 
GO

-- =============================================
-- Author:		Tim Cooke
-- Create date: 23/01/2017
-- Description:	Gets all supplier sites that should be notified for a check
-- Version:		1
-- =============================================
/*
    Example call:
    SELECT * FROM [dbo].[FNC_ResponseHeader_SupplierSiteID]()
*/
-- =============================================
CREATE FUNCTION [dbo].[FNC_ResponseHeader_SupplierSiteID] ()
RETURNS TABLE
AS
RETURN

    SELECT R.ResponseID
         , RH.Header_SourceID AS LocationID
      FROM QMS_Response R
      JOIN QMS_ResponseHeader RH ON R.ResponseID = RH.ResponseID
      JOIN QMS_AssessmentHeader AH ON RH.AssessmentHeaderID = AH.AssessmentHeaderID
     WHERE R.RecordStatus < 201
       AND RH.Header_SourceID IS NOT NULL
       AND AH.DataTypeID = 9
       AND AH.DataLookupID = 53
       AND AH.RecordStatus < 201

GO
-- END 04_FNC_ResponseHeader_SupplierSiteID


-- BEGIN 05_FNC_Response_UserList
IF NULLIF(OBJECT_ID('dbo.FNC_Response_UserList'), 0) IS NOT NULL DROP FUNCTION [dbo].[FNC_Response_UserList] 
GO
-- =============================================
-- Author:		Markp
-- Create date: 16/09/2009
-- Description:	Gets all users that should be notified for a check
-- Version:		6
-- =============================================
-- JE updated 10/01/2012 - Added filter by Business Group
-- JT TFS 4194 - 14/08/2012 - when @StockCodeID is null, we don't want to filter out users with 
--							  stockcode/sub-group/business unit permissions
-- MP TFS 6422 - fix to handle supplier site permissions
-- JE TFS 8956: alerts no longer go to all Supplier users when no Supplier captured in header
-- TC Re-written to make more performant for larger data sets.
/*
    Example call:
    SELECT * FROM [dbo].[FNC_Response_UserList](336) ORDER BY UserId
*/
-- JT TFS 95216: suppliersite permissions being picked up in error
-- =============================================
CREATE FUNCTION [dbo].[FNC_Response_UserList] 
(
	@ResponseID int 
)
RETURNS TABLE 
AS
RETURN
    SELECT DISTINCT U.UserId
      FROM QMS_User U
     WHERE 1=1
       
       --Get stock code from response header, consignment item and delivery item 
       AND (NOT EXISTS (SELECT * 
                         FROM VW_PermissionLink PL1 
                        WHERE U.UserID = PL1.UserID 
                          AND PL1.PermissionAllow = 1 
                          AND PL1.StockCodeID IS NOT NULL 
                          AND PL1.PermissionType IN (101,102,103,104,105,106) 
                          --Only do this if we have a stock code
                          AND (SELECT COUNT(*) 
                                 FROM dbo.FNC_ResponseHeader_StockCodeID() A 
                                WHERE A.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL2 
                          INNER JOIN dbo.FNC_ResponseHeader_StockCodeID() STK 
                               ON PL2.StockCodeID = STK.StockCodeID 
                               AND STK.ResponseId = @ResponseId 
                               AND PL2.PermissionType = 104 --Stock Code Permission Type
                               AND PL2.UserID = U.UserID))
      
       --Get location from response header
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL3 
                         WHERE U.UserID = PL3.UserID 
                           AND PL3.PermissionAllow = 1 
                           AND PL3.LocationID IS NOT NULL 
                           AND PL3.PermissionType IN (1, 101,102,103,104,105,106) 
                           --Only do this if we have a location
                           AND (SELECT COUNT(*) 
                                  FROM dbo.FNC_ResponseHeader_LocationID() B 
                                 WHERE B.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL4 
                          JOIN dbo.FNC_ResponseHeader_LocationID() LOC 
                               ON PL4.LocationID = LOC.LocationID 
                               AND LOC.ResponseId = @ResponseId 
                               AND PL4.PermissionType = 103 --Location Permission Type
                               AND PL4.UserID = U.UserID))
        
       --Get supplier from response header
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL5 
                         WHERE U.UserID = PL5.UserID 
                           AND PL5.PermissionAllow = 1 
                           AND PL5.SupplierID IS NOT NULL 
                           AND PL5.PermissionType IN (101,102,103,104,105,106))
                           --No restriction for Supplier to exist in the header to accomodate TFS 8956.
                           --All Supplier users aren't notified when no Supplier captured
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL6 
                          JOIN dbo.FNC_ResponseHeader_SupplierID() SUP 
                               ON PL6.SupplierID = SUP.SupplierID 
                               AND SUP.ResponseId = @ResponseId 
                               AND PL6.PermissionType = 101 --Supplier Permission Type
                               AND PL6.UserID = U.UserID))
     
       --Get customer from response header
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL7 
                         WHERE U.UserID = PL7.UserID 
                           AND PL7.PermissionAllow = 1 
                           AND PL7.CustomerID IS NOT NULL 
                           AND PL7.PermissionType IN (101,102,103,104,105,106) 
                           --Only do this if we have a customer
                           AND (SELECT COUNT(*) 
                                  FROM dbo.FNC_ResponseHeader_CustomerID() D 
                                 WHERE D.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL8 
                          JOIN dbo.FNC_ResponseHeader_CustomerID() CUS 
                               ON PL8.CustomerID = CUS.CustomerID 
                               AND CUS.ResponseId = @ResponseId 
                               AND PL8.PermissionType = 102 --Customer Permission Type
                               AND PL8.UserID = U.UserID))
      
       --Get supplier site from response header
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL9 
                         WHERE U.UserID = PL9.UserID 
                           AND PL9.PermissionAllow = 1 
                           AND PL9.LocationID IS NOT NULL 
                           AND PL9.PermissionType IN (101,102,103,104,105,106) 
                           --Only do this if we have a supplier
                           AND (SELECT COUNT(*) 
                                  FROM dbo.FNC_ResponseHeader_SupplierSiteID() E 
                                 WHERE E.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL10 
                          JOIN dbo.FNC_ResponseHeader_SupplierSiteID() SUPS 
                               ON PL10.LocationID = SUPS.LocationID 
                               AND SUPS.ResponseId = @ResponseId 
                               AND PL10.PermissionType = 103 --Supplier Site Permission Type
                               AND PL10.UserID = U.UserID))
         
       --Get customer assigned to stock code
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL11 
                         WHERE U.UserID = PL11.UserID 
                           AND PL11.PermissionAllow = 1 
                           AND PL11.CustomerID IS NOT NULL 
                           --Only do this if we have a customer
                           AND (SELECT COUNT(*) 
                                  FROM dbo.FNC_ResponseHeader_CustomerID() F 
                                 WHERE F.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL12 
                          JOIN PM_VW_StockCode SC1 
                               ON U.UserID = PL12.UserID 
                               AND PL12.PermissionType = 102 --Customer Permission Type (again)
                               AND PL12.CustomerID = SC1.CustomerID 
                               AND EXISTS(SELECT * 
                                            FROM dbo.FNC_ResponseHeader_StockCodeID() STK2 
                                           WHERE SC1.StockCodeID = STK2.StockCodeID 
                                            AND STK2.ResponseId = @ResponseId)))

       --accomodate TFS 2565 by joining to PM_VW_StockCode then onto FNC_ResponseHeader_StockCodeID function
       -----------------------------------------------------------------------------------------------------------------
       --Get Business Group from Stock Code
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL13 
                         WHERE U.UserID = PL13.UserID 
                           AND PL13.PermissionAllow = 1 
                           AND PL13.BusinessGroup IS NOT NULL 
                           --Only do this if we have a Business Group
                           AND (SELECT COUNT(*) 
                                  FROM dbo.FNC_ResponseHeader_StockCodeID() F 
                                 WHERE F.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL14 
                          INNER MERGE JOIN PM_VW_StockCode SC2 
                               ON U.UserID = PL14.UserID 
                               AND PL14.PermissionType = 106 --Business Group Permission Type
                               AND PL14.BusinessGroup = SC2.BusinessGroup 
                               AND EXISTS(SELECT * 
                                            FROM dbo.FNC_ResponseHeader_StockCodeID() STK3 
                                           WHERE SC2.StockCodeID = STK3.StockCodeID 
                                             AND STK3.ResponseId = @ResponseId)))
 
       --Get Stock Code Sub Group from Stock Code
       AND (NOT EXISTS (SELECT * 
                          FROM VW_PermissionLink PL15 
                         WHERE U.UserID = PL15.UserID 
                           AND PL15.PermissionAllow = 1 
                           AND PL15.StockCodeSubGroup IS NOT NULL 
                           --Only do this if we have a Stock Code Sub Group
                           AND (SELECT COUNT(*) 
                                  FROM dbo.FNC_ResponseHeader_StockCodeID() G 
                                 WHERE G.ResponseId = @ResponseId) > 0)
             OR EXISTS (SELECT * 
                          FROM VW_PermissionLink PL16 
                          JOIN PM_VW_StockCode SC3 
                               ON U.UserID = PL16.UserID 
                               AND PL16.PermissionType = 105 --Stock Code Sub Group Permission Type
                               AND PL16.StockCodeSubGroup = SC3.StockCodeSubGroup 
                               AND EXISTS (SELECT * 
                                             FROM dbo.FNC_ResponseHeader_StockCodeID() STK4 
                                            WHERE SC3.StockCodeID = STK4.StockCodeID 
                                              AND STK4.ResponseId = @ResponseId)))

       -----------------------------------------------------------------------------------------------------------------
       AND U.LeftCompany = 0
       AND U.RecordStatus < 201

GO
-- END 05_FNC_Response_UserList


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which ResponseIDs need to be put in the filter.
--              Can be used by any report.
-- Version:		23
-- =============================================
-- j.toolan 30-mar-2012: If @AssessmentTypeID_LIST is empty, then join all assessments
-- j.toolan 13 Apr 2012: Fixed bug with 2nd filter incorrectly clearing.
-- j.toolan 18 Apr 2012: Added Code to filter by stockcode where appropriate (renamed to ReportFilter)
-- JE 25 Apr 2012: made ReportFilterID1 optional, so can just filter by AssessmentType if necessary (TFS 3382)
-- j.toolan 6 July 2012: TFS 4013: Dynamic filtering added.
-- JE TFS 4740: removed GROUP BY for performance reasons
-- JE TFS 5037: corrected @ReportFilterID2 check as 2nd tier filter wasn't working
-- MP TFS 5289: changed date format used for custom filter functions
-- JT TFS 5396: changed code to allow commas in FilterTexts passed in as CSV. (replace "," with "&comma;" )
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JT TFS 8117: Added offline filtering.
-- MP TFS 8227: Made temp table user specific and added performance logging
-- JE TFS 8274: Put result set of FNC_User_ResponseList into temp table before using it
-- JT TFS 13-667: Do not filter responses if the date range is too big.
-- JT TFS 13-1076: Search either QMS_Response or QMS_SupplierResponse, but not both.
-- JF TFS 2281 - if filter doesn't have a sourceid, this wasn't working
-- JT TFS 13-17467: Only include checks which are eligible candidates for the data warehouse.
-- JT TFS 13-38578: Convert SP to use QRY_User_ResponseList (rather than FNC_UserResponseList)
-- JT TFS 13-40008: 22/02/2016 - Added transaction IDs to response list generation, and extracted stockcode filtering
-- JT TFS 13-42222: 02/03/2016 - Added link to TokenUserID so that reports can be run simultaneously.
-- JF TFS 35438 - make sure we remove data which wouldn't show in the data warehouse - ie parent checks for Assessment type 5
-- jf TFS 90761 - Morrisons QC Office - Issue with reports not generating - removed the use of FNC_Response_ShouldIncludeInDataWarehouse dues to performance issues 
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST nvarchar(4000),
	@FilterText_LIST nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID2 int = NULL,
	@FilterText2 nvarchar(255) = NULL,
	
	@Offline bit = NULL,
	@SearchLocalChecksIfOffline bit = 1, -- Use QMS_Response even if offline (default)
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	EXEC SAVE_TEMP_Transaction @TransactionID, @UserID

	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	IF @Offline IS NULL SELECT @Offline = CASE WHEN db_name() = 'GreenlightQC_Mobile' THEN 1 ELSE 0 END
	
	DECLARE @SearchOffline bit -- if this is set, then we search in QMS_SupplierResponse.
	SELECT @SearchOffline  = CASE WHEN @Offline = 1 AND @SearchLocalChecksIfOffline = 0 THEN 1 ELSE 0 END
		
	-- Do not filter responses if the date range is not specified
	DECLARE @performResponseFiltering bit 
	SELECT @performResponseFiltering = CASE WHEN @DateFrom <= '1 Jan 2001' and datediff(year, @DateFrom, @DateTo) >= 10 THEN 0 ELSE 1 END
	
	DELETE TEMP_ResponseFilter WHERE UserID = @TokenUserID OR TransactionID = @TransactionID

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(@FilterID_LIST) = 0 AND LEN(@FilterText_LIST) = 0 -- first criteria is empty?!
	AND @ReportFilterID2 IS NOT NULL AND (@FilterID2 IS NOT NULL OR @FilterText2 IS NOT NULL)) -- but the second is not...
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST = CONVERT(nvarchar(4000), @FilterID2),
				@FilterText_LIST = @FilterText2
				
		SELECT @ReportFilterID2 = NULL, @FilterID2 = NULL, @FilterText2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF @ReportFilterID2 IS NOT NULL AND (NULLIF(@FilterID2, 0) IS NULL) AND (LEN(ISNULL(@FilterText2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))
	INSERT @FilterItem (pos, id, txt)
	
	-- tfs 2281 - if filter doesn't have a sourceidm this wasn't working
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 
	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END

	-- Do some custom filtering if it is set up. This acts like "AND" (restricting results)
	DECLARE @CustomFilterFunction nvarchar(50)
	SET @CustomFilterFunction = NULL
	SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
	FROM APP_ReportFilter 
	WHERE ReportFilterID = @ReportFilterID1
	
	DECLARE @customSql nvarchar(1000)
	SET @customSql = NULL
	CREATE TABLE #tmpCustomFilter (UserID INT, ResponseID int, GroupByText nvarchar(50))

	DECLARE @FilteredResponse TABLE (id int not null)
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN
		DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		IF @performResponseFiltering = 1
		BEGIN
			-- either the custom function runs, *OR* the generic one, *OR* the offline one, but not more than one.
			IF @SearchOffline = 1
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID) + ', ' +
									'''' + convert(nvarchar(50), @FilterText) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'
				
				PRINT @customSql
				
				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
			END
			ELSE
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
		END
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- insert all our new custom filtered items (added to temp table in loop above)
	INSERT @FilteredResponse (id)
	SELECT ResponseID 
	FROM #tmpCustomFilter
	WHERE UserID = @TokenUserID

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN 
		IF @performResponseFiltering = 1
		BEGIN
			DECLARE @SecondFilteredResponse TABLE (id int)
			
			SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
			FROM APP_ReportFilter 
			WHERE ReportFilterID = @ReportFilterID2
		
			IF @SearchOffline = 1
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END 
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				DELETE #tmpCustomFilter WHERE UserID = @TokenUserID --clear it out for second filter.
				
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID2) + ', ' +
									'''' + convert(nvarchar(50), @FilterText2) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'

				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
				
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID FROM #tmpCustomFilter WHERE UserID = @TokenUserID
			END
			ELSE
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response (@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END
			
			DELETE @FilteredResponse 
			WHERE id NOT IN (
				SELECT id FROM @SecondFilteredResponse
			)
		END
	END

	-- Remove all responses which are not suitable candidates for the data warehouse
	-- JF - logic moved further down as @FilteredResponse is not used in all return paths
	--DELETE @FilteredResponse
	--WHERE dbo.FNC_Response_ShouldIncludeInDataWarehouse(id) = 0


	IF @performResponseFiltering = 1
	BEGIN
		IF @SearchOffline = 1
		BEGIN
			IF @ReportFilterID1 IS NOT NULL
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY r.ResponseID
			ELSE
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id	
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY r.ResponseID
		END
		ELSE
		BEGIN
			--JE TFS 82774 - temp table of responses user can view (moved from direct access in query to improve performance)
			IF nullif(object_id('tempdb..#tmpUserResponses'), 0) IS NOT NULL DROP TABLE #tmpUserResponses
			CREATE TABLE #tmpUserResponses (UserID int, ResponseID int, ResponseStatus int, QAFlagID int)
			INSERT #tmpUserResponses (UserID, ResponseID, ResponseStatus, QAFlagID)
			EXEC QRY_User_ResponseList @UserID, @DateFrom, @DateTo
			
			IF @ReportFilterID1 IS NOT NULL
				-- Only return responses that the user is allowed to see, filter by Assessment Type, and use filter
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE rl.UserID = @UserID
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY rl.ResponseID
			
			IF @ReportFilterID1 IS NULL
				-- Only return responses that the user is allowed to see, but there has been no filter specified, so only filter by Assessment Type (if specified)
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				WHERE rl.UserID = @UserID	
				--AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				and (
					SELECT	CASE	WHEN APP_AssessmentType.AssessmentFormat NOT IN (5,6,7) THEN 1 -- Isn't multi-check at all - store it.
							WHEN APP_AssessmentType.AssessmentFormat = 5
								 AND  QMS_Response.ParentResponseGUID IS NOT NULL	THEN 1 -- Morrisons scenario - only store the child checks.
							WHEN APP_AssessmentType.AssessmentFormat = 6 
								 AND  QMS_Response.ParentResponseGUID IS NULL		THEN 1 -- barfoots scenario - only store the parent check.
							WHEN APP_AssessmentType.AssessmentFormat = 7			THEN 1 -- standard multi-check - store both child and parent. 
							ELSE 0 -- Reject it.
					END
					FROM	QMS_Response
					INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID
					WHERE	QMS_Response.ResponseID = r.ResponseID
				) =1

				GROUP BY rl.ResponseID
			
			DROP TABLE #tmpUserResponses
		END

	END
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST = ' + CASE WHEN LEN(@FilterID_LIST)=0 THEN 'NULL' ELSE @FilterID_LIST END
			SET @ReportString = @ReportString + ', @FilterText_LIST = ' + CASE WHEN LEN(@FilterText_LIST)=0 THEN 'NULL' ELSE @FilterText_LIST END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID2 = ' + CAST(@FilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterText2 = ' + CASE WHEN LEN(@FilterText2)=0 THEN 'NULL' ELSE @FilterText2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Response_PreviousList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Response_PreviousList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 28/07/2008
-- Description:	Gets list of previous responses
-- Version:		28
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- Jonathan Toolan: 26-Mar-2012 Location Added.
-- Jonathan Toolan: TFS 3211 Multiple AssessmentTypeID_LIST
-- JE TFS 6272 - added Child Check/Top Level Check functionality
-- JT TFS 7346 - Added AssessmentID filter parameter.
-- JT TFS 7989 - Added ExportSetupTable to this SP as an example, so that custom SPs can copy and modify more easily.
-- JT TFS 8130 - ShowOutstanding to switch between "outstanding" mode and "completed" mode.
-- JT TFS 8578 - Added generic filtering if applied.
-- JE TFS 13-466: added provision for AQC Multi assessment format
-- JT TFS 13-612: Added DisplayMode parameter to show multi-check children (default) or parents.
-- JE TFS 13-1621: amended setting of @multiCheckDisplay so parent checks only are shown when isViewQCChecksScreen
-- JE TFS 13-2537: amended so that 3:isQCActionsScreen shows Parent checks only
-- JF TFS 2880 - Ensure that Child checks are available in Office Web - View QC Checks. 
				--Basically if the Multicheck contains a Summary column, then I would expect the Parent check to be referenced/displayed in ALL views
				--If no Summary Column (majority of multicheck customers) - then the Child check should be referenced/displayed in ALL views
-- TFS 2955		Navigate to Tasks > QC Actions for an assessment type of Active Quality Chain,Expected outcome: I am expecting to see the Parent checks here, Actual outcome: I am seeing the Child Checks
-- JE TFS 13-9699: only show non-Approved child checks for AQC assessment type
-- JE TFS 13-11892: output column indicating what sort of multi check response  is (i.e. parent or child) for AQC
-- JE TFS 13-12540: AQC Multi Check shows parent checks only for Previous tab
-- MP TFS 13-12735: Added logic to check for AQC approval config setting
-- jf 13042 - AQC may not be multicheck so allow for it
-- PH TFS 13-15140:  Added 'IsWebChecks' column to determine whether the Response is authorised to be edited using WebChecks front end (permission derived from APP_Menu with ParentID = 233)
-- jf tfs 16880 -- removed inner joins to FNC_User_ResponseList and used TEMP_selected instead - UBE Check view performance for user with 2000+ stock code permissions
				-- took resource string calcs out of nexted select and put in main output query (QAFlagID, ResourceName,QAFlag)
-- PH TFS 13-34304:		Reformatted and refactored QRY_Response_PreviousList to make more sense, perform better and be forward compatible
-- PH TFS 13-34304:		Reformatted to close up gaps
-- SS TFS 13-33017: When AssessmentTypeID = 16, AQC_FNC_ResponseHeader is now used instead of FNC_ResponseHeader
-- JT TFS 13-82047: Extract SP for shelf life to improve performance.
-- JF TFS 13-89956: It appears the origanl issue with using FNC_ResponseHeader for AQC checks has been resolved so we no longer have to use AQC_FNC_ResponseHeader (which performs badly)
-- JF TFS 13-89956: replaced multiple calls to FNC_Response_IsMultiCheck with TEMP_selected value
-- =============================================
ALTER PROCEDURE [dbo].[QRY_Response_PreviousList]
--DECLARE
    @AssessmentTypeID INT = NULL ,
    @AssessmentTypeID_LIST NVARCHAR(255) = NULL ,
    @DateFrom DATETIME ,
    @DateTo DATETIME ,
    @UserID INT ,
    @AssessmentID INT = NULL ,
    @ShowOutstanding BIT = 0 , -- 0: show completed checks only, 1: show outstanding checks only, NULL: show all checks regardless of ResponseStatus.
    @IsFilterApplied BIT = 0 ,
    @DisplayMode INT = NULL	
		-- 0 or NULL: not specified.
		-- 1:isPreviousChecksScreen
		-- 2:isViewQCChecksScreen
		-- 3:isQCActionsScreen
		-- 4:isQCCheckReport

		--SELECT  @AssessmentTypeID = 6,  @DateFrom = '1 Aug 2017 00:00:00',  @DateTo = '15 Sep 2017 23:59:59',  @UserID = 1,  @ShowOutstanding = 1,  @DisplayMode = 1
AS 
BEGIN
    SET NOCOUNT ON
--exec QRY_Response_PreviousList @AssessmentTypeID=13,@AssessmentTypeID_LIST=default,@DateFrom='2015-02-19 00:00:00',@DateTo='2015-03-19 23:59:59',@UserID=100,@AssessmentID=default,@ShowOutstanding=NULL,@IsFilterApplied=0,@DisplayMode=2    
-- If (isPreviousChecksScreen), then show parent checks (1), otherwise show children (2).
-- @multiCheckDisplay should match return value from dbo.FNC_Response_IsMultiCheck.

--FNC_Response_IsMultiCheck results
--0 -- Isn't multi-check at all.
--1 -- Is a parent multi-check
--2 -- Is a child multi-check
																				
        

--get Assessment Format
    DECLARE @AssessmentFormat int;
    SELECT  @AssessmentFormat = AssessmentFormat
    FROM    APP_AssessmentType
    WHERE   AssessmentTypeID = @AssessmentTypeID;


	DECLARE @multiCheckDisplay int;
    SELECT  @multiCheckDisplay = CASE WHEN ISNULL(@DisplayMode, 0) IN ( 1 ) 
									THEN  1    --show parent checks only for previouschecks screen
									WHEN @AssessmentFormat = 6 THEN 1
									ELSE 2
								END;
									
	DECLARE @AQCApproval bit;
	SELECT @AQCApproval = ISNULL(dbo.FNC_GetOptionValue(120),0);

    IF (@AssessmentTypeID_LIST IS NULL)
	BEGIN 
        SET @AssessmentTypeID_LIST = CONVERT(NVARCHAR, @AssessmentTypeID);
	END

--Clear temp table
    DELETE  FROM TEMP_Selected
    WHERE   ( UserID = @UserID );

--Fill temp table
    IF (@IsFilterApplied = 0)
    BEGIN
        INSERT INTO TEMP_Selected
			(
				SelectedID,
				UserID
			)
		SELECT	QMS_Response.ResponseID,
				@UserID AS UserID
		FROM	QMS_Response
		INNER JOIN	dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList
						ON QMS_Response.ResponseID = User_ResponseList.ResponseID
		INNER JOIN	dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',') AS SplitString
			ON QMS_Response.AssessmentTypeID = SplitString.Token;
    END
    ELSE 
    BEGIN
        INSERT INTO TEMP_Selected
                (
					SelectedID,
					UserID
				)
		SELECT	ResponseID,
				@UserID AS UserID
		FROM	TEMP_ResponseFilter
		WHERE	(UserID = @UserID);

    END

	UPDATE TEMP_Selected SET SelectedValue = dbo.FNC_Response_IsMultiCheck(SelectedID ) WHERE (UserID = @UserID);
    
	SET NOCOUNT OFF

    IF (@AssessmentTypeID_LIST = '2') -- Arrival check -------------------------------------------------------------------------------------------------------------------------------------
    BEGIN
        SELECT	QMS_Response.ResponseID,
				Attachments.NumAttachments,
				QMS_Assessment.AssessmentName,
				ResponseHeader.Supplier,
				ResponseHeader.StockCode, 
                ResponseHeader.Location,
				QMS_Response.StartDate,
				QMS_Response.EndDate,
				QMS_Response.Actual_Score,
				QMS_Response.Actual_QAFlagID,
				ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) as QAFlag,
                QMS_Response.Override_QAFlagID, 
				ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID),
				QAFlag_Override.QAFlag) as Override_QAFlag,
				QMS_Response.Offline_ResponseID, 
                CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete,
				dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck,
				QMS_User.FullName
		FROM    QMS_Response
		INNER JOIN	TEMP_Selected
					ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
					AND TEMP_Selected.UserID = @UserID
		INNER JOIN	QMS_Assessment
				ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID
		INNER JOIN	QMS_User
				ON QMS_Response.UserID = QMS_User.UserID
		LEFT OUTER JOIN dbo.FNC_ResponseHeader() AS ResponseHeader
				ON QMS_Response.ResponseID = ResponseHeader.ResponseID
		LEFT OUTER JOIN
                    (
						SELECT	SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
						FROM	QMS_Attachment
						WHERE	AttachmentSourceID = 1
								AND RecordStatus < 201
						GROUP BY SourceObjectID
					) AS Attachments
					ON QMS_Response.ResponseID = Attachments.ResponseID
		LEFT OUTER JOIN
                    (
						SELECT	QAFlagID, ResourceName,QAFlag
						FROM	APP_QAFlag
					) AS QAFlag_Override
					ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID
		LEFT OUTER JOIN
                    (
						SELECT	QAFlagID, ResourceName,QAFlag
						FROM	APP_QAFlag AS APP_QAFlag_1
					) AS QAFlag
					ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
		WHERE	
			(
				QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)
				AND 
				(
					@AssessmentID = QMS_Response.AssessmentID
					OR @AssessmentID IS NULL
				)
				--AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)
				AND TEMP_Selected.SelectedValue IN (0, @multiCheckDisplay)
			)
			AND 
			(
				@ShowOutstanding IS NULL
				OR
				(
					@ShowOutstanding = 1
					AND QMS_Response.ResponseStatus = 0
				)
				OR
				(
					@ShowOutstanding = 0
					AND QMS_Response.ResponseStatus <> 0
				)
			)    
		ORDER BY QMS_Response.EndDate DESC;
    END
    ELSE
	BEGIN 
        IF (@AssessmentTypeID_LIST IN ( '3', '4', '5' )) -- Online/Label/Dispatch -------------------------------------------------------------------------------------------------------------------------------------
        BEGIN
            SELECT	QMS_Response.ResponseID,
					Attachments.NumAttachments,
					ResponseHeader.Customer,
					ResponseHeader.StockCode,
					ResponseHeader.Location, 
					ResponseHeader.DepotDate AS [Depot date],
					QMS_Assessment.AssessmentName,
					QMS_Response.StartDate,
					QMS_Response.EndDate, 
					QMS_Response.Actual_QAFlagID, 
					ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) AS QAFlag,
					QMS_Response.Override_QAFlagID, 
					ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) AS Override_QAFlag,
					QMS_Response.Offline_ResponseID,
					CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete,
					dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
					QMS_User.FullName
			FROM	QMS_Response
			INNER JOIN	TEMP_Selected
				ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
				AND TEMP_Selected.UserID = @UserID
			INNER JOIN	QMS_Assessment
				ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID
			INNER JOIN	QMS_User
				ON QMS_Response.UserID = QMS_User.UserID
			LEFT OUTER JOIN	dbo.FNC_ResponseHeader() AS ResponseHeader
				ON QMS_Response.ResponseID = ResponseHeader.ResponseID
			LEFT OUTER JOIN
                    (
						SELECT	SourceObjectID AS ResponseID,
								COUNT(AttachmentID) AS NumAttachments
						FROM	QMS_Attachment
						WHERE   AttachmentSourceID = 1
								AND RecordStatus < 201
						GROUP BY SourceObjectID
					) AS Attachments
					ON QMS_Response.ResponseID = Attachments.ResponseID
			LEFT OUTER JOIN
                    (
						SELECT	QAFlagID,
								ResourceName,
								QAFlag
						FROM	APP_QAFlag
					) AS QAFlag_Override
					ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID
			LEFT OUTER JOIN
                    (
						SELECT	QAFlagID,
								ResourceName,
								QAFlag
						FROM	APP_QAFlag
					) AS QAFlag
					ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
			WHERE
				(
					QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)
					AND
					(
						@AssessmentID = QMS_Response.AssessmentID
						OR @AssessmentID IS NULL
					)
					--AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)
					AND TEMP_Selected.SelectedValue IN (0, @multiCheckDisplay)
				)
				AND
				(
					@ShowOutstanding IS NULL
					OR
					(
						@ShowOutstanding = 1
						AND	QMS_Response.ResponseStatus = 0
					)
					OR
					(
						@ShowOutstanding = 0
						AND QMS_Response.ResponseStatus <> 0
					)
				)
			ORDER BY QMS_Response.EndDate DESC;
        END
        ELSE IF (@AssessmentTypeID_LIST = '6') -- Shelf life -------------------------------------------------------------------------------------------------------------------------------------
        BEGIN
			--SELECT 
			--		UserID = @UserID, 
			--		AssessmentID = @AssessmentID,
			--		AssessmentTypeID_LIST = @AssessmentTypeID_LIST,
			--		ShowOutstanding = @ShowOutstanding,
			--		multiCheckDisplay = @multiCheckDisplay

		     EXEC QRY_Response_PreviousList_ShelfLife 
					@UserID = @UserID, 
					@AssessmentID = @AssessmentID,
					@AssessmentTypeID_LIST = @AssessmentTypeID_LIST,
					@ShowOutstanding = @ShowOutstanding,
					@multiCheckDisplay = @multiCheckDisplay
        END

		ELSE IF (@AssessmentTypeID_LIST = '16') --Active Quality Chain -------------------------------------------------------------------------------------------------------------------------------------
        BEGIN
            SELECT	QMS_Response.ResponseID,
					Attachments.NumAttachments,
					ResponseHeader.Customer,
					ResponseHeader.StockCode,
					ResponseHeader.Location, 
					ResponseHeader.AssessmentSubType AS [Test type],
					ResponseHeader.DepotDate AS [Depot date],
					QMS_Assessment.AssessmentName,
					QMS_Response.StartDate, 
					QMS_Response.EndDate,
					QMS_Response.Actual_QAFlagID,
					ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) AS QAFlag,
					QMS_Response.Override_QAFlagID, 
					ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID),
					QAFlag_Override.QAFlag) AS Override_QAFlag,
					QMS_Response.Offline_ResponseID,
					CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete,
					dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
					QMS_User.FullName, CAST(CASE WHEN @AQCApproval = 1 THEN ISNULL(dbo.FNC_Response_IsApproved(QMS_Response.ResponseID), 0) ELSE 1 END AS BIT) AS IsApproved,
					--ISNULL(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID), 0) AS MultiCheckLevel
					ISNULL(TEMP_Selected.SelectedValue,0) AS MultiCheckLevel
			FROM	QMS_Response
			INNER JOIN	TEMP_Selected
				ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
				AND TEMP_Selected.UserID = @UserID
			INNER JOIN	QMS_Assessment
				ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID
			INNER JOIN	QMS_User ON QMS_Response.UserID = QMS_User.UserID
			--LEFT OUTER JOIN	dbo.AQC_FNC_ResponseHeader() AS ResponseHeader
			LEFT OUTER JOIN	dbo.FNC_ResponseHeader() AS ResponseHeader
				ON QMS_Response.ResponseID = ResponseHeader.ResponseID
				
			LEFT OUTER JOIN
					(
						SELECT	SourceObjectID AS ResponseID,
								COUNT(AttachmentID) AS NumAttachments

						FROM	QMS_Attachment

						WHERE	AttachmentSourceID = 1
								AND RecordStatus < 201

						GROUP BY SourceObjectID
					) AS Attachments
					ON QMS_Response.ResponseID = Attachments.ResponseID 
			LEFT OUTER JOIN
					(
						SELECT	QAFlagID,
								ResourceName,
								QAFlag
						FROM	APP_QAFlag
					) AS QAFlag_Override
					ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID
					
			LEFT OUTER JOIN
					(
						SELECT	QAFlagID,
								ResourceName,
								QAFlag

						FROM	APP_QAFlag
					) AS QAFlag
					ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
			WHERE
				(
					QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)
					AND
					(
						(
							@multiCheckDisplay = 1
							--AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1
							AND TEMP_Selected.SelectedValue=1
						)
						OR
						(
							@multiCheckDisplay = 2
							--AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)
							AND TEMP_Selected.SelectedValue IN (1,2)
						)
						--OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0
						OR TEMP_Selected.SelectedValue = 0
					)
					AND
					(
						@AssessmentID = QMS_Response.AssessmentID
						OR @AssessmentID IS NULL
					)
				)
				AND
				(
					@ShowOutstanding IS NULL
					OR
					(
						@ShowOutstanding = 1
						AND QMS_Response.ResponseStatus = 0
					)
					OR
					(
						@ShowOutstanding = 0
						AND QMS_Response.ResponseStatus <> 0
					)
				)
			ORDER BY QMS_Response.EndDate DESC;
        END
        ELSE 
        BEGIN -------------------------------------------------------------------------------------------------------------------------------------
            SELECT	QMS_Response.ResponseID,
					Attachments.NumAttachments,
					QMS_Assessment.AssessmentName,
					ResponseHeader.Location,
					QMS_Response.StartDate, 
					QMS_Response.EndDate,
					QMS_Response.Actual_Score,
					QMS_Response.Actual_QAFlagID, 
					ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) AS QAFlag, 							  
					QMS_Response.Override_QAFlagID, 
					ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) AS Override_QAFlag,
					QMS_Response.Offline_ResponseID,
					CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) 
					= 0 THEN 0 ELSE 1 END AS BIT) AS Complete,
					dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck,
					QMS_User.FullName
			FROM	QMS_Response
			INNER JOIN	TEMP_Selected
				ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
				AND TEMP_Selected.UserID = @UserID
			INNER JOIN	QMS_Assessment
				ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID
			INNER JOIN	QMS_User
				ON QMS_Response.UserID = QMS_User.UserID
			LEFT OUTER JOIN
					(
						SELECT	SourceObjectID AS ResponseID,
								COUNT(AttachmentID) AS NumAttachments
						FROM	QMS_Attachment
						WHERE	AttachmentSourceID = 1
								AND RecordStatus < 201
						GROUP BY SourceObjectID
					) AS Attachments
					ON QMS_Response.ResponseID = Attachments.ResponseID
			LEFT OUTER JOIN
					(
						SELECT	QAFlagID,
								ResourceName,
								QAFlag
						FROM	APP_QAFlag
					) AS QAFlag_Override
					ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID
			LEFT OUTER JOIN
					(
						SELECT	QAFlagID,
								QAFlag,ResourceName 
						FROM	APP_QAFlag
					) AS QAFlag
					ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
			LEFT OUTER JOIN	dbo.FNC_ResponseHeader() AS ResponseHeader
					ON QMS_Response.ResponseID = ResponseHeader.ResponseID
			WHERE
				(
					QMS_Response.AssessmentTypeID IN
							(
								SELECT	CONVERT(INT, Token) AS Expr1
								FROM	dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
							)
					AND
					(
						@AssessmentID = QMS_Response.AssessmentID OR
						@AssessmentID IS NULL
					)
					AND
					(
						--dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)
						TEMP_Selected.SelectedValue IN (0, @multiCheckDisplay)
					) 
				)
				AND
				(
					@ShowOutstanding IS NULL
					OR
					(
						@ShowOutstanding = 1
						AND QMS_Response.ResponseStatus = 0
					)
					OR
					(
						@ShowOutstanding = 0
						AND QMS_Response.ResponseStatus <> 0
					)
				)
			ORDER BY QMS_Response.EndDate DESC;
        END
	END

--get list of columns that need to be hidden in Excel export
--NOTE:		tableindex is important, and zero based. This export table is table 1 because it's the second table returned in the dataset
--		FixedColumns: These are returned in the main body. These should be fixed. Leave this as NULL to use default ones specified by front end.
--		StaticColumns: These are displayed underneath the compact text in the grid, and do not move, but are specific for each customer.
--		CyclicColumns: If this contains the word "REMAINING_COLUMNS", then it will also include any columns not specified in Fixed or Static.
-- NOTE:	leabing static and cyclic columns NULL will use first two columns not in the default fixed columns as "static", and other columns not in either fixed or static as "cyclic"
    SELECT	1 AS TableIndex,
            CONVERT(BIT, 1) AS ExportSetupTable ,
            'Portrait/Landscape' AS Orientation ,
            'n/a' AS SheetName ,
            'n/a' AS HiddenColumns ,
            'n/a' AS FixedColumns ,
            'n/a' AS StaticColumns ,
            'n/a' AS CyclicColumns
    UNION

    SELECT  0 ,
            0 ,
            'Portrait' ,
            'View Checks' ,
            NULL ,
            NULL ,
            'Location,FullName' ,
            'REMAINING_COLUMNS'
    ORDER BY TableIndex ASC

END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'UP_User_Password')
	EXEC ('CREATE PROCEDURE [dbo].[UP_User_Password] AS SELECT ''stub version, to be replaced''')
GO
-- ===================================================
-- Author		: Jon Edwards
-- Create date	: 23/05/2011
-- Description	: Updates password when changed at login
-- Version		: 8 
-- ===================================================
-- J.Toolan 8 Mar 2012: Added SHA2 password field to update
-- JT TFS 6629 - Expire old userpassword entries before adding a new one.
-- JT TFS 6791 - Handle cases where there are more than one username in the QMS_User table (and exclude deleted people)
-- Martin P TFS 13 - 42414 - Changed any GTEDATE() to GETUTCDATE()
-- PH TFS 13-66014 : v6 03/02/2017 - Now only storing single password and must use HashType.
-- PH TFS 13-66163 : v7 03/02/2017 - Make extraneous password columns null for row, if password updated.
-- MDS TFS 91512 : v8 23/01/2018 - Set expiry date 50 years in future for AQC suppliers.
-- ===================================================
ALTER PROCEDURE [dbo].[UP_User_Password]
	@UserName nvarchar(255),
	@Password nvarchar(255),
	@HashType int
AS
BEGIN

	SET NOCOUNT ON;

	-- #tmpUser should only contain one username, but historically it sometimes has more.
	IF NULLIF(object_id('tempdb..#tmpUser'), 0) IS NOT NULL DROP TABLE #tmpUser
	SELECT UserID
	INTO #tmpUser
	FROM QMS_User
	WHERE UserName = @UserName 
	AND RecordStatus < 201
	
	-- update User table
	UPDATE QMS_User
	SET
		[Password] = @Password,
		Password_MD5 = NULL,
		Password_SHA2 = NULL,
		HashType = @HashType,
        Updated = GETUTCDATE()
	WHERE UserID IN (SELECT UserID FROM #tmpUser)
	
	UPDATE QMS_UserPassword 
	SET ExpiryDate = getutcdate()
	WHERE UserID IN (SELECT UserID FROM #tmpUser)
	AND ISNULL(ExpiryDate, '31 Dec 9999') > getutcdate()
	
	-- add new row to passwords table
	INSERT INTO QMS_UserPassword
	(
		
		UserID,
		Password,
		CreationDate,
		ExpiryDate,
		HashType
	)
	SELECT
		UserID,
		@Password,
		GETUTCDATE(),
		ExpiryDate = 
			CASE 
				WHEN [dbo].[FNC_GetUserLevel](UserID) = 10 THEN DATEADD(YEAR, 50, GETDATE()) --AQC users
			ELSE 
				DATEADD(day, CAST(dbo.FNC_GetOptionValue(46) AS int), GETUTCDATE())
			END,
		@HashType
	FROM QMS_User
	WHERE UserID IN (SELECT UserID FROM #tmpUser)
END

GO



PRINT 'Database updated successfully!'
GO

