-- =============================================
-- Author:		Jeremy Formby
-- Create date: 3/05/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON

GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_User_ResponseList')
	EXEC('CREATE FUNCTION [dbo].[FNC_User_ResponseList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 17/08/2009
-- Description:	Returns list of responses that user can access
-- Version: 	17
-- =============================================
-- JT TFS 4194 - 14/08/2012 - When stock code is restricted, this now returns checks where no stock code is specified in the header.
-- JE TFS 4622 - returns QAFlagID based on ReportQAFlag config setting
-- JT TFS 5801 - performance improvements
-- JE TFS 5905 - updated to make use of Reporting Warehouse table
-- MP TFS 6031 - changed non-admin filter to ensure that temp checks are not included
-- MP TFS 6152 - fix to handle supplier site permissions
-- JT TFS 6689 - FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7612 - Void checks visibility is controlled by UserSetting for specific reports.
-- JE TFS 13-3570: added PRIMARY KEY to @Table_StockCode to improve performance
-- MP TFS 13-12783: fixed issue with ambiguous Location permission linking
-- jf tfs 16880 - UBE Check view performance for user with 2000+ stock code permissions - added lots of filters for updates like ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)
-- MP TFS 23409 - Added conditions in the CanUseWarehouse block so that Suppliers (userlevel=9) use supplier site code and non supplier (userlevel<>9) use locations
-- SS TFS 13-49452: Added logic so that when a client is using datawarehousing, if they are filtering on today's date, the warehouse is used for all dates up to today's date AND then today's date data
-- is worked out without the warehouse being used. This prevents timeouts FROM happening.
-- SS TFS 13-49897: Added ISNULL(Uploaded, EndDate) so that EndDate will be used if Uploaded contains a NULL Value
-- PH TSF 13-61027 : v16 18/11/2016 - Bug Fix.  When including checks from the datawarehouse for todays date, an assumption was made that the datawarehouse had been updated before @DateFrom.  
--                   However, if this was not the case then checks were included from the past (but after the warehouse had been updated).  The greater of the datawarehouse update time and @DateFrom is now used.
-- JT TFS 13-82627 : Fix issue of Uploaded results appearing on QC Checks screen not in the date range specified.
-- =============================================
ALTER FUNCTION [dbo].[FNC_User_ResponseList]
(
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	ResponseStatus tinyint,
	QAFlagID int
)
AS
BEGIN

--declare @UserID int, @DateFrom datetime,	@DateTo datetime
--select @UserID = 836, @DateFrom = '1 Jun 2013 00:00:00',  @DateTo = '21 May 2014 23:59:59'

	DECLARE @UnfilteredResponse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	-- This will contain responses which are not in the warehouse, typically those with today's date.
	DECLARE @UnfilteredResponseNotInWarehouse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	--get date reporting warehouse table last built
	DECLARE @WarehouseBuildDate datetime
	SET @WarehouseBuildDate = (SELECT Updated FROM APP_Config WHERE OptionID = 78)

	DECLARE @UserSettingDisplayVoids bit 
	SET @UserSettingDisplayVoids = ISNULL(cast(dbo.FNC_GetUserSetting(@UserID, 'DisplayVoidChecks') as bit), 0)

	-- Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserSettingDisplayVoids = 1 -- Include all recordstatuses when we set a report to do the same by setting the User Setting.
		SET @RecordStatus = 255 -- maximum possible. 
	ELSE IF @UserLevel = 1  --Admin can always see hidden/void checks.
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 100

	-- Permissions that this user has.
	DECLARE @PermissionLink TABLE (
		UserID int, SupplierID int, CustomerID int, LocationID int, StockCodeID int, 
		StockCodeSubGroup nvarchar(100), BusinessGroup nvarchar(100)
	)
	INSERT INTO	@PermissionLink (UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup)
	SELECT	UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup
	FROM	VW_PermissionLink PermissionLink
	WHERE	PermissionLink.PermissionAllow = 1 AND PermissionLink.PermissionType IN (101,102,103,104,105,106) AND UserID = @UserID

	DECLARE @Table_StockCode TABLE (
		StockCodeID int PRIMARY KEY
	)
	
	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code, and translate to StockCodeID
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		BusinessGroup IN (SELECT BusinessGroup FROM @PermissionLink WHERE BusinessGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @PermissionLink WHERE StockCodeSubGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeID IN (SELECT StockCodeID FROM @PermissionLink WHERE StockCodeID IS NOT NULL)

	-- The datawarehouse can still be used when the user filters after the last warehouse build, but it will not contain much (if any) useful checks.
	-- We insert all the recent checks into @UnfilteredResponse so that the datawarehouse can still be used when the user filters on today's date.
	-- Just because the date range in entirely contained within the warehouse, doesn't mean that we only search in the warehouse. 
	-- Checks uploaded after the last warehouse build, but whose EndDate is within that range need to be included, as do incomplete checks (ResponseStatus 0)
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1
	BEGIN
		DECLARE @override bit 
		SET @override = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

		DECLARE @ResponseFromDate datetime
		SELECT @ResponseFromDate = CASE WHEN @WarehouseBuildDate > @DateFrom THEN @WarehouseBuildDate ELSE @DateFrom END
			
		-- Get the majority of checks which will not be in the warehouse - the ones added after the warehouse has been built.
		INSERT INTO @UnfilteredResponseNotInWarehouse (ResponseID, ResponseStatus, QAFlagID) 
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM	QMS_Response
		WHERE	(RecordStatus < @RecordStatus) 
		AND EndDate BETWEEN @ResponseFromDate AND @DateTo


		-- Make sure we include the checks uploaded after the warehouse was last built, but within the entire date range. They will not yet be in the warehouse.
		-- Also, we have to make sure that incomplete checks get included, since they are never put in the warehouse.
		INSERT INTO @UnfilteredResponseNotInWarehouse (ResponseID, ResponseStatus, QAFlagID) 
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM QMS_Response
		WHERE	(RecordStatus < @RecordStatus) 
		AND (Uploaded > @WarehouseBuildDate OR ResponseStatus = 0)
		AND EndDate BETWEEN @DateFrom AND @DateTo
		AND ResponseID NOT IN (SELECT ResponseID FROM @UnfilteredResponseNotInWarehouse)


		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponseTodaysDate
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
			ISNULL(UnfilteredResponseTodaysDate.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponseTodaysDate
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
			ISNULL(UnfilteredResponseTodaysDate.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponseTodaysDate
			SET		SupplierID = QMS_ResponseHeader.Header_SourceID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
			ISNULL(UnfilteredResponseTodaysDate.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponseTodaysDate
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponseTodaysDate.ResponseID = HeaderStockCode.ResponseID INNER JOIN
						PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponseTodaysDate.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponseTodaysDate
			SET		StockCodeID = HeaderStockCode.StockCodeID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponseTodaysDate.ResponseID = HeaderStockCode.ResponseID
			WHERE ISNULL(UnfilteredResponseTodaysDate.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)


		-- Include checks from the warehouse.
		INSERT INTO	@UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
		SELECT		RPT_Warehouse_Response.ResponseID, QMS_Response.ResponseStatus, RPT_Warehouse_Response.QAFlagID --Actual or Override QAFlag setting checked when building warehouse table
		FROM		RPT_Warehouse_Response 
		INNER JOIN  QMS_Response ON RPT_Warehouse_Response.ResponseID = QMS_Response.ResponseID
		WHERE		(QMS_Response.RecordStatus < @RecordStatus) 
		AND (RPT_Warehouse_Response.EndDate BETWEEN @DateFrom AND @WarehouseBuildDate)
		AND NOT EXISTS (SELECT UnfilteredResponseTodaysDate.ResponseID FROM @UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate
						WHERE UnfilteredResponseTodaysDate.ResponseID = RPT_Warehouse_Response.ResponseID)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponse
			SET	LocationID = RPT_Warehouse_Response.SupplierSiteID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.SupplierSiteID,0)
 
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponse
			SET	LocationID = RPT_Warehouse_Response.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponse
			SET		SupplierID = RPT_Warehouse_Response.SupplierID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(RPT_Warehouse_Response.SupplierID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponse
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID INNER JOIN
						PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponse
			SET		StockCodeID = RPT_Warehouse_Response.StockCodeID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID

		INSERT INTO @UnfilteredResponse(ResponseID, ResponseStatus, LocationID, CustomerID, SupplierID, StockCodeID, QAFlagID)
		SELECT ResponseID, ResponseStatus, LocationID, CustomerID, SupplierID, StockCodeID, QAFlagID FROM @UnfilteredResponseNotInWarehouse
	END
	ELSE --don't use warehouse table
	BEGIN
		DECLARE @OverrideOption bit 
		SET @OverrideOption = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
			
		INSERT INTO @UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @OverrideOption = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM	QMS_Response
		WHERE	(RecordStatus < @RecordStatus) AND (EndDate BETWEEN @DateFrom AND @DateTo)
			
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponse
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
			ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponse
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
			ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponse
			SET		SupplierID = QMS_ResponseHeader.Header_SourceID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
			ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponse
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID INNER JOIN
						PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponse
			SET		StockCodeID = HeaderStockCode.StockCodeID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID
			WHERE ISNULL(UnfilteredResponse.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)
	END

	-- Only return responses which match the filter specified, or for which we do not have a filter.
	-- This means that admin users (any user with no permissions set) can see all responses.
	INSERT INTO @Response (ResponseID, ResponseStatus, QAFlagID)
	SELECT	ResponseID, ResponseStatus, QAFlagID
	FROM	@UnfilteredResponse UnfilteredResponse
	WHERE (	NOT EXISTS (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL)
		OR	LocationID IN (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
		OR	SupplierID IN (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL)
		OR	CustomerID IN (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL) )
	AND (	StockCodeID IS NULL -- Allow unlinked stockcode responses
		OR	NOT EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		OR	StockCodeID IN (SELECT StockCodeID FROM @Table_StockCode) )
	
	RETURN
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which ResponseIDs need to be put in the filter.
--              Can be used by any report.
-- Version:		22
-- =============================================
-- j.toolan 30-mar-2012: If @AssessmentTypeID_LIST is empty, then join all assessments
-- j.toolan 13 Apr 2012: Fixed bug with 2nd filter incorrectly clearing.
-- j.toolan 18 Apr 2012: Added Code to filter by stockcode where appropriate (renamed to ReportFilter)
-- JE 25 Apr 2012: made ReportFilterID1 optional, so can just filter by AssessmentType if necessary (TFS 3382)
-- j.toolan 6 July 2012: TFS 4013: Dynamic filtering added.
-- JE TFS 4740: removed GROUP BY for performance reasons
-- JE TFS 5037: corrected @ReportFilterID2 check as 2nd tier filter wasn't working
-- MP TFS 5289: changed date format used for custom filter functions
-- JT TFS 5396: changed code to allow commas in FilterTexts passed in as CSV. (replace "," with "&comma;" )
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JT TFS 8117: Added offline filtering.
-- MP TFS 8227: Made temp table user specific and added performance logging
-- JE TFS 8274: Put result set of FNC_User_ResponseList into temp table before using it
-- JT TFS 13-667: Do not filter responses if the date range is too big.
-- JT TFS 13-1076: Search either QMS_Response or QMS_SupplierResponse, but not both.
-- JF TFS 2281 - if filter doesn't have a sourceid, this wasn't working
-- JT TFS 13-17467: Only include checks which are eligible candidates for the data warehouse.
-- JT TFS 13-38578: Convert SP to use QRY_User_ResponseList (rather than FNC_UserResponseList)
-- JT TFS 13-40008: 22/02/2016 - Added transaction IDs to response list generation, and extracted stockcode filtering
-- JT TFS 13-42222: 02/03/2016 - Added link to TokenUserID so that reports can be run simultaneously.
-- JF TFS 35438 - make sure we remove data which wouldn't show in the data warehouse - ie parent checks for Assessment type 5
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST nvarchar(4000),
	@FilterText_LIST nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID2 int = NULL,
	@FilterText2 nvarchar(255) = NULL,
	
	@Offline bit = NULL,
	@SearchLocalChecksIfOffline bit = 1, -- Use QMS_Response even if offline (default)
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	EXEC SAVE_TEMP_Transaction @TransactionID, @UserID

	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	IF @Offline IS NULL SELECT @Offline = CASE WHEN db_name() = 'GreenlightQC_Mobile' THEN 1 ELSE 0 END
	
	DECLARE @SearchOffline bit -- if this is set, then we search in QMS_SupplierResponse.
	SELECT @SearchOffline  = CASE WHEN @Offline = 1 AND @SearchLocalChecksIfOffline = 0 THEN 1 ELSE 0 END
		
	-- Do not filter responses if the date range is not specified
	DECLARE @performResponseFiltering bit 
	SELECT @performResponseFiltering = CASE WHEN @DateFrom <= '1 Jan 2001' and datediff(year, @DateFrom, @DateTo) >= 10 THEN 0 ELSE 1 END
	
	DELETE TEMP_ResponseFilter WHERE UserID = @TokenUserID OR TransactionID = @TransactionID

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(@FilterID_LIST) = 0 AND LEN(@FilterText_LIST) = 0 -- first criteria is empty?!
	AND @ReportFilterID2 IS NOT NULL AND (@FilterID2 IS NOT NULL OR @FilterText2 IS NOT NULL)) -- but the second is not...
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST = CONVERT(nvarchar(4000), @FilterID2),
				@FilterText_LIST = @FilterText2
				
		SELECT @ReportFilterID2 = NULL, @FilterID2 = NULL, @FilterText2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF @ReportFilterID2 IS NOT NULL AND (NULLIF(@FilterID2, 0) IS NULL) AND (LEN(ISNULL(@FilterText2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))
	INSERT @FilterItem (pos, id, txt)
	
	-- tfs 2281 - if filter doesn't have a sourceidm this wasn't working
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 
	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END

	-- Do some custom filtering if it is set up. This acts like "AND" (restricting results)
	DECLARE @CustomFilterFunction nvarchar(50)
	SET @CustomFilterFunction = NULL
	SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
	FROM APP_ReportFilter 
	WHERE ReportFilterID = @ReportFilterID1
	
	DECLARE @customSql nvarchar(1000)
	SET @customSql = NULL
	CREATE TABLE #tmpCustomFilter (UserID INT, ResponseID int, GroupByText nvarchar(50))

	DECLARE @FilteredResponse TABLE (id int not null)
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN
		DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		IF @performResponseFiltering = 1
		BEGIN
			-- either the custom function runs, *OR* the generic one, *OR* the offline one, but not more than one.
			IF @SearchOffline = 1
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID) + ', ' +
									'''' + convert(nvarchar(50), @FilterText) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'
				
				PRINT @customSql
				
				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
			END
			ELSE
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
		END
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- insert all our new custom filtered items (added to temp table in loop above)
	INSERT @FilteredResponse (id)
	SELECT ResponseID 
	FROM #tmpCustomFilter
	WHERE UserID = @TokenUserID

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN 
		IF @performResponseFiltering = 1
		BEGIN
			DECLARE @SecondFilteredResponse TABLE (id int)
			
			SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
			FROM APP_ReportFilter 
			WHERE ReportFilterID = @ReportFilterID2
		
			IF @SearchOffline = 1
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END 
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				DELETE #tmpCustomFilter WHERE UserID = @TokenUserID --clear it out for second filter.
				
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID2) + ', ' +
									'''' + convert(nvarchar(50), @FilterText2) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'

				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
				
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID FROM #tmpCustomFilter WHERE UserID = @TokenUserID
			END
			ELSE
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response (@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END
			
			DELETE @FilteredResponse 
			WHERE id NOT IN (
				SELECT id FROM @SecondFilteredResponse
			)
		END
	END

	-- Remove all responses which are not suitable candidates for the data warehouse
	-- JF - logic moved further down as @FilteredResponse is not used in all return paths
	--DELETE @FilteredResponse
	--WHERE dbo.FNC_Response_ShouldIncludeInDataWarehouse(id) = 0


	IF @performResponseFiltering = 1
	BEGIN
		IF @SearchOffline = 1
		BEGIN
			IF @ReportFilterID1 IS NOT NULL
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				GROUP BY r.ResponseID
			ELSE
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id	
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				GROUP BY r.ResponseID
		END
		ELSE
		BEGIN
			--JE TFS 82774 - temp table of responses user can view (moved from direct access in query to improve performance)
			IF nullif(object_id('tempdb..#tmpUserResponses'), 0) IS NOT NULL DROP TABLE #tmpUserResponses
			CREATE TABLE #tmpUserResponses (UserID int, ResponseID int, ResponseStatus int, QAFlagID int)
			INSERT #tmpUserResponses (UserID, ResponseID, ResponseStatus, QAFlagID)
			EXEC QRY_User_ResponseList @UserID, @DateFrom, @DateTo
			
			IF @ReportFilterID1 IS NOT NULL
				-- Only return responses that the user is allowed to see, filter by Assessment Type, and use filter
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE rl.UserID = @UserID
				AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				GROUP BY rl.ResponseID
			
			IF @ReportFilterID1 IS NULL
				-- Only return responses that the user is allowed to see, but there has been no filter specified, so only filter by Assessment Type (if specified)
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				WHERE rl.UserID = @UserID	
				AND dbo.FNC_Response_ShouldIncludeInDataWarehouse(r.ResponseID) = 1
				GROUP BY rl.ResponseID
			
			DROP TABLE #tmpUserResponses
		END

	END
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST = ' + CASE WHEN LEN(@FilterID_LIST)=0 THEN 'NULL' ELSE @FilterID_LIST END
			SET @ReportString = @ReportString + ', @FilterText_LIST = ' + CASE WHEN LEN(@FilterText_LIST)=0 THEN 'NULL' ELSE @FilterText_LIST END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID2 = ' + CAST(@FilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterText2 = ' + CASE WHEN LEN(@FilterText2)=0 THEN 'NULL' ELSE @FilterText2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SYNC_Import_PM_GoodsIntake')
	EXEC ('CREATE PROCEDURE [dbo].[SYNC_Import_PM_GoodsIntake] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		unknown
-- Create date: 18/12/2008
-- Description:	imports data into the mobile PM_GoodsIntake table
-- Version:		7
-- =============================================
-- jf tfs 18253 - added @PalletStockCodeID
-- jf tfs 29101 - added storeid
-- jf TFS 23605 - make sure no date conversions happen
-- jf tfs 32575 - add @CountryID and @Alias
-- JF TFS 53109 - Added CustomerID
-- JF TFS 71972 - Added 10 lookup columns
-- JF TFS 86848 - allow intake insert if @ConsignmentItemID is null
-- =============================================
ALTER PROCEDURE [dbo].[SYNC_Import_PM_GoodsIntake] 
	@GoodsIntakeID int,
	@SupplierID int,
	@ConsignmentItemID int,
	@LocationID int,
	@ItemCount real,
	@UnitQuantity int,
	@PalletNo int,
	@NetWeight real = NULL,
	@DeliveryID int,
	@DateArrived datetime,
	@SupplierRef nvarchar(50),
	@Barcode nvarchar(50),
	@OriginalBarcode nvarchar(50) = NULL,
	@UserGroupID int,
	@HarvestDate datetime,
	@SupplierLotNo nvarchar(50),
	@Comments nvarchar(255),
	@UnitStockCodeID int = NULL,
	@E_Text1 nvarchar(50) = NULL,
	@E_Text2 nvarchar(50) = NULL,
	@E_Numeric1 int = NULL,
	@E_Numeric2 int = NULL,
	@E_DateTime1 datetime = NULL,
	@PalletID INT = NULL,
	@Certified BIT = NULL,
	@ArrivalTemp REAL = NULL,
	@E_Numeric3 REAL = NULL,
	@Updated DATETIME = NULL,
	@RecordStatus tinyint,
	@GUID uniqueidentifier,
	@Created bigint,
	@RowVersion timestamp,
	@PalletStockCodeID int,
	@StoreID int = null,
	@CountryID int = NULL,
	@Alias nvarchar(50) = NULL,
	@CustomerID int = null,
	@LookupText1 nvarchar(100) = NULL, 
	@LookupText2 nvarchar(100) = NULL, 
	@LookupText3 nvarchar(100) = NULL, 
	@LookupText4 nvarchar(100) = NULL, 
	@LookupText5 nvarchar(100) = NULL, 
	@LookupText6 nvarchar(100) = NULL, 
	@LookupText7 nvarchar(100) = NULL, 
	@LookupText8 nvarchar(100) = NULL, 
	@LookupText9 nvarchar(100) = NULL, 
	@LookupText10 nvarchar(100) = NULL,

	@InsertCount int = 0 OUTPUT,
	@UpdateCount int = 0 OUTPUT,
	@ReceivedAnchor bigint = 0 OUTPUT
AS

SET @DateArrived = dbo.FNC_ConvertServerDate(@DateArrived)
SET @HarvestDate = dbo.FNC_ConvertServerDate(@HarvestDate)
SET @E_DateTime1 = dbo.FNC_ConvertServerDate(@E_DateTime1)

IF @Certified IS NULL
	SET @Certified = 0
	
IF NOT EXISTS(SELECT * FROM dbo.PM_GoodsIntake WHERE (GoodsIntakeID = @GoodsIntakeID))
BEGIN --Append record
	IF @RecordStatus < 201 AND (EXISTS(SELECT ConsignmentID FROM dbo.PM_ConsignmentItems WHERE (ConsignmentItemID = @ConsignmentItemID)) OR @ConsignmentItemID IS NULL)
	BEGIN
		INSERT INTO dbo.PM_GoodsIntake
                      (GoodsIntakeID, SupplierID, ConsignmentItemID, LocationID, ItemCount, UnitQuantity, PalletNo, NetWeight, DeliveryID, DateArrived, SupplierRef, SupplierLotNo, Comments, 
                      UserGroupID, HarvestDate, Barcode, UnitStockCodeID, OriginalBarcode, E_Text1, E_Text2, E_Numeric1, E_Numeric2, E_DateTime1, PalletID, Certified, ArrivalTemp, 
                      E_Numeric3, Updated, RecordStatus, GUID,PalletStockCodeID,StoreID, CountryID, Alias, CustomerID,LookupText1, LookupText2, LookupText3, LookupText4, LookupText5, LookupText6, LookupText7, LookupText8, LookupText9, LookupText10)
		VALUES     (@GoodsIntakeID,@SupplierID,@ConsignmentItemID,@LocationID,@ItemCount,@UnitQuantity,@PalletNo,@Netweight,@DeliveryID,@DateArrived,@SupplierRef,@SupplierLotNo,@Comments,@UserGroupID,@HarvestDate,@Barcode,@UnitStockCodeID,@OriginalBarcode,@E_Text1,@E_Text2,@E_Numeric1,@E_Numeric2,@E_DateTime1,@PalletID,@Certified,@ArrivalTemp,@E_Numeric3,@Updated,@RecordStatus,@GUID,@PalletStockCodeID,@StoreID, @CountryID, @Alias, @CustomerID,@LookupText1, @LookupText2, @LookupText3, @LookupText4, @LookupText5, @LookupText6, @LookupText7, @LookupText8, @LookupText9, @LookupText10)
		SET @InsertCount = @@rowcount
	END
END

ELSE
BEGIN --Update record
	UPDATE    dbo.PM_GoodsIntake
	SET              SupplierID = @SupplierID, ConsignmentItemID = @ConsignmentItemID, LocationID = @LocationID, DeliveryID = @DeliveryID, ItemCount = @ItemCount, 
                      UnitQuantity = @UnitQuantity, PalletNo = @PalletNo, NetWeight = @Netweight, DateArrived = @DateArrived, SupplierRef = @SupplierRef, Barcode = @Barcode, 
                      SupplierLotNo = @SupplierLotNo, Comments = @Comments, UserGroupID = @UserGroupID, HarvestDate = @HarvestDate, UnitStockCodeID = @UnitStockCodeID, 
                      OriginalBarcode = @OriginalBarcode, E_Text1 = @E_Text1, E_Text2 = @E_Text2, E_Numeric1 = @E_Numeric1, E_Numeric2 = @E_Numeric2, 
                      E_DateTime1 = @E_DateTime1, PalletID = @PalletID, Certified = @Certified, ArrivalTemp = @ArrivalTemp, E_Numeric3 = @E_Numeric3, Updated = @Updated, 
                      RecordStatus = @RecordStatus, PalletStockCodeID=@PalletStockCodeID,StoreID=@StoreID,CountryID=@CountryID,Alias=@Alias,
					  CustomerID=@CustomerID,
					  LookupText1=@LookupText1, 
					  LookupText2=@LookupText2, 
					  LookupText3=@LookupText3, 
					  LookupText4=@LookupText4, 
					  LookupText5=@LookupText5, 
					  LookupText6=@LookupText6, 
					  LookupText7=@LookupText7, 
					  LookupText8=@LookupText8, 
					  LookupText9=@LookupText9, 
					  LookupText10=@LookupText10
	WHERE     (GoodsIntakeID = @GoodsIntakeID)
	SET @UpdateCount = @@rowcount
END

IF CAST(@RowVersion AS bigint) > @Created
	SET @ReceivedAnchor = CAST(@RowVersion AS bigint)
ELSE
	SET @ReceivedAnchor = @Created
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Activity_CA_Failed')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Activity_CA_Failed] AS SELECT ''stub version, to be replaced''')
GO
-- ===================================================    
-- Author:  Jonathan Toolan    
-- Create date: 22/11/2013    
-- Description: All failed corrective actions in the last week assigned to the user.    
-- Version:  4    
-- ===================================================    
-- JT TFS 8093 - Created    
-- JT TFS 8369 - Modified to show assigned to supplier and groups.    
-- JE TFS 13-12641: remove use of table valued functions and replace with join to QMS_CorrectiveActionUser    
-- Martin P TFS 13-69273: 22/03/2017- Added @TranslateStrings parameter and added new strings for i18n        
-- ===================================================    
ALTER PROCEDURE [dbo].[QRY_Activity_CA_Failed]    
 @UserID int,    
 @CurrentDate datetime = NULL,    
 @TranslateStrings BIT = 0     
AS    
BEGIN    
 SET NOCOUNT ON    
 IF @CurrentDate IS NULL SELECT @CurrentDate = dbo.FNC_GetDate()    
 SET NOCOUNT OFF    
    
 SELECT   
 CASE WHEN @TranslateStrings = 1 THEN  
  '[[[Corrective Action]]] ' + CASE WHEN QMS_CorrectiveAction.CAStatusID = 6 THEN '[[[Failed]]]' ELSE '[[[Overdue]]]' END   
 ELSE   
  'Corrective Action ' + CASE WHEN QMS_CorrectiveAction.CAStatusID = 6 THEN 'Failed' ELSE 'Overdue' END  
 END AS FlagText,  
 CASE WHEN @TranslateStrings = 1 THEN  
  '<strong>${Priority}</strong> [[[priority Corrective Action for]]] <strong>${CATitle}</strong> ${DueOrRejected} [[[on]]] ${EventDate}<br />${CASubTitle}: ${CAReason}'   
 ELSE  
  '<strong>${Priority}</strong> priority Corrective Action for <strong>${CATitle}</strong> ${DueOrRejected} on ${EventDate}<br />${CASubTitle}: ${CAReason}'   
 END AS DisplayText,  
 EventDate = CASE WHEN QMS_CorrectiveAction.CAStatusID = 6 THEN QMS_CorrectiveAction.DateFailed ELSE QMS_CorrectiveAction.DateDeadline END,    
 CASE WHEN QMS_CorrectiveAction.CAStatusID = 6 THEN 'rejected' ELSE 'due' END  AS DueOrRejected,  
 QMS_CorrectiveAction.CorrectiveActionID,    
 QMS_CorrectiveAction.CATitle,    
 QMS_CorrectiveAction.CAReason,    
 QMS_CorrectiveAction.DateDeadline,    
 QMS_CorrectiveAction.AssignUserID,    
  ISNULL(NULLIF(QMS_CorrectiveAction.CASubTitle, ''), 'Comments') AS CASubTitle,  
 CASE WHEN @TranslateStrings = 1 THEN  
  ISNULL(( SELECT TOP 1 PriorityLevel FROM APP_PriorityLevel WHERE PriorityLevelID = QMS_CorrectiveAction.PriorityLevelID), 'Unknown')    
 ELSE  
  ISNULL(( SELECT TOP 1 CASE WHEN ResourceName IS NOT NULL THEN '${' + ResourceName + '}'    
     ELSE PriorityLevel  END     
  FROM APP_PriorityLevel WHERE PriorityLevelID = QMS_CorrectiveAction.PriorityLevelID), 'Unknown')    
 END AS Priority  
 FROM QMS_CorrectiveAction     
 INNER JOIN QMS_CorrectiveActionUser ON QMS_CorrectiveAction.CorrectiveActionID = QMS_CorrectiveActionUser.CorrectiveActionID    
 WHERE (QMS_CorrectiveAction.CAStatusID IN (3, 6)) -- Failed (because of deadline) or rejected    
 AND (QMS_CorrectiveAction.RecordStatus < 201)    
 AND (QMS_CorrectiveAction.DateDeadline > dateadd(week, -1, @CurrentDate))    
 AND (QMS_CorrectiveActionUser.UserID = @UserID)    
END 
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_SupplierRef_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_SupplierRef_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 17/11/2008
-- Description:	Gets lookup of Supplier Ref used (AWB for Flamingo)
-- version:	3
-- =============================================
-- JF TFS 74073 - allow supplier ref to be picked up from goods intake barcode
-- MDS TFS 82742 - don't allow DateArrived parameter to be used if filter is not enabled
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_SupplierRef_Combo] 
	@SupplierID int = 0,
	@DateArrived datetime = NULL,
	@Barcode nvarchar(50) = null
AS

if @Barcode IS NOT NULL and (SELECT Enabled FROM dbo.APP_DataLookupTrigger WHERE DataLookupID_Trigger = 11 AND DataLookupID_Update = 15) = 1
	SELECT    SupplierRef 
	FROM         dbo.PM_VW_Goodsintake 
	WHERE PM_VW_Goodsintake.Barcode = @Barcode

else IF (@DateArrived IS NULL or 
		(@DateArrived IS NOT NULL and (SELECT Enabled FROM dbo.APP_DataLookupTrigger WHERE DataLookupID_Trigger = 11 AND DataLookupID_Update = 15) = 0))
	SELECT DISTINCT SupplierRef 
	FROM         dbo.PM_VW_GoodsIntake
	WHERE     (SupplierRef IS NOT NULL) AND (@SupplierID <= 0) OR
						  (SupplierRef IS NOT NULL) AND (@SupplierID > 0) AND (SupplierID = @SupplierID)
	ORDER BY SupplierRef

ELSE
	SELECT DISTINCT SupplierRef
	FROM         dbo.PM_VW_GoodsIntake
	WHERE     (SupplierRef IS NOT NULL) AND (@SupplierID <= 0) AND (dbo.FNC_TrimDate(DateArrived) = dbo.FNC_TrimDate(@DateArrived)) OR
						  (SupplierRef IS NOT NULL) AND (@SupplierID > 0) AND (SupplierID = @SupplierID) AND (dbo.FNC_TrimDate(DateArrived) 
						  = dbo.FNC_TrimDate(@DateArrived))
	ORDER BY SupplierRef
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_Intake_E_Text2_Lookup')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_Intake_E_Text2_Lookup] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JMF
-- Create date: 1/1/2017
-- Description:	Get E_text2 for intake
-- Version:	1
-- =============================================
-- JF TFS 82724 - pick up e-text2 from Intake barcode
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_Intake_E_Text2_Lookup] 
	@Barcode nvarchar(50)=null
AS

	SELECT     Distinct E_Text2 
	FROM         dbo.PM_VW_Goodsintake
	WHERE     (Barcode  = @Barcode and @Barcode IS NOT NULL)
GO



PRINT 'Database updated successfully!'
GO

