-- =============================================
-- Author:		Jeremy Formby
-- Create date: 19/03/2015
-- Version:		0.1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 59
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO


IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'[dbo].[QMS_UserPermissions]') and name='IX_QMS_UserPermissions_PermissionTitle')
BEGIN TRY
	create index IX_QMS_UserPermissions_PermissionTitle on QMS_UserPermissions(UserID,PermissionAllow,PermissionType)
	include(PermissionTitle);


	DBCC DBREINDEX ('QMS_UserPermissions')
	
END TRY 
BEGIN CATCH

	INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Creating index IX_QMS_UserPermissions_PermissionTitle'
	FROM #tmpScriptName
	
END CATCH
GO


IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'[dbo].[APP_ResourceLocale]') and name='IX_APP_ResourceLocale_CountryLocaleID_ResourceID')
BEGIN TRY
	create index IX_APP_ResourceLocale_CountryLocaleID_ResourceID on APP_ResourceLocale(CountryLocaleID,ResourceID);

	DBCC DBREINDEX ('APP_ResourceLocale')
	
END TRY 
BEGIN CATCH

	INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Creating index IX_APP_ResourceLocale_CountryLocaleID_ResourceID'
	FROM #tmpScriptName
	
END CATCH
GO


IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'[dbo].[APP_Resource]') and name='IX_APP_Resource_ResourceName')
BEGIN TRY
	create index IX_APP_Resource_ResourceName on APP_Resource(ResourceName);

	DBCC DBREINDEX ('APP_Resource')
	
END TRY 
BEGIN CATCH

	INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Creating index IX_APP_Resource_ResourceName'
	FROM #tmpScriptName
	
END CATCH
GO


-- BEGIN [dbo].[IX_QMS_CorrectiveAction_SourceObjectID]
IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'[dbo].[QMS_CorrectiveAction]') and name='IX_QMS_CorrectiveAction_SourceObjectID')
BEGIN 
	CREATE NONCLUSTERED INDEX [IX_QMS_CorrectiveAction_SourceObjectID] ON [dbo].[QMS_CorrectiveAction]
	(
		[SourceObjectID] ASC
	)
	INCLUDE ([CorrectiveActionID]) 

	DBCC DBREINDEX ('QMS_CorrectiveAction')
END
GO
-- END [dbo].[IX_QMS_CorrectiveAction_SourceObjectID]

--PRINT 'Inserting tables...'

--PRINT 'Updating tables...'

--PRINT 'Creating indexes...'

--PRINT 'Updating Views...'

--PRINT 'Dropping Views...'

--PRINT 'Updating User Defined Functions...'
--GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_User_ResponseList')
	EXEC('CREATE FUNCTION [dbo].[FNC_User_ResponseList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 17/08/2009
-- Description:	Returns list of responses that user can access
-- Version: 	12
-- =============================================
-- JT TFS 4194 - 14/08/2012 - When stock code is restricted, this now returns checks where no stock code is specified in the header.
-- JE TFS 4622 - returns QAFlagID based on ReportQAFlag config setting
-- JT TFS 5801 - performance improvements
-- JE TFS 5905 - updated to make use of Reporting Warehouse table
-- MP TFS 6031 - changed non-admin filter to ensure that temp checks are not included
-- MP TFS 6152 - fix to handle supplier site permissions
-- JT TFS 6689 - FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7612 - Void checks visibility is controlled by UserSetting for specific reports.
-- JE TFS 13-3570: added PRIMARY KEY to @Table_StockCode to improve performance
-- MP TFS 13-12783: fixed issue with ambiguous Location permission linking
-- jf tfs 16880 - UBE Check view performance for user with 2000+ stock code permissions - added lots of filters for updates like ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)
-- =============================================
ALTER FUNCTION FNC_User_ResponseList
(
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	ResponseStatus tinyint,
	QAFlagID int
)
AS
BEGIN

--declare @UserID int, @DateFrom datetime,	@DateTo datetime
--select @UserID = 836, @DateFrom = '1 Jun 2013 00:00:00',  @DateTo = '21 May 2014 23:59:59'

	DECLARE @UnfilteredResponse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	--get date reporting warehouse table last built
	DECLARE @WarehouseBuildDate datetime
	SET @WarehouseBuildDate = (SELECT Updated FROM APP_Config WHERE OptionID = 78)

	DECLARE @UserSettingDisplayVoids bit 
	SET @UserSettingDisplayVoids = ISNULL(cast(dbo.FNC_GetUserSetting(@UserID, 'DisplayVoidChecks') as bit), 0)

	-- Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserSettingDisplayVoids = 1 -- Include all recordstatuses when we set a report to do the same by setting the User Setting.
		SET @RecordStatus = 255 -- maximum possible. 
	ELSE IF @UserLevel = 1  --Admin can always see hidden/void checks.
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 100

	-- Permissions that this user has.
	DECLARE @PermissionLink TABLE (
		UserID int, SupplierID int, CustomerID int, LocationID int, StockCodeID int, 
		StockCodeSubGroup nvarchar(100), BusinessGroup nvarchar(100)
	)
	INSERT INTO	@PermissionLink (UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup)
	SELECT	UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup
	FROM	VW_PermissionLink PermissionLink
	WHERE	PermissionLink.PermissionAllow = 1 AND PermissionLink.PermissionType IN (101,102,103,104,105,106) AND UserID = @UserID

	DECLARE @Table_StockCode TABLE (
		StockCodeID int PRIMARY KEY
	)
	
	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code, and translate to StockCodeID
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		BusinessGroup IN (SELECT BusinessGroup FROM @PermissionLink WHERE BusinessGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @PermissionLink WHERE StockCodeSubGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeID IN (SELECT StockCodeID FROM @PermissionLink WHERE StockCodeID IS NOT NULL)

	BEGIN
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 AND @DateTo <= @WarehouseBuildDate --use warehouse table if built within specified date range
		BEGIN
			INSERT INTO	@UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
			SELECT		RPT_Warehouse_Response.ResponseID, QMS_Response.ResponseStatus, RPT_Warehouse_Response.QAFlagID --Actual or Override QAFlag setting checked when building warehouse table
			FROM		RPT_Warehouse_Response INNER JOIN
							QMS_Response ON RPT_Warehouse_Response.ResponseID = QMS_Response.ResponseID
			WHERE		(QMS_Response.RecordStatus < @RecordStatus) AND (RPT_Warehouse_Response.EndDate BETWEEN @DateFrom AND @DateTo)
			
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL)
				UPDATE	UnfilteredResponse 
				SET		LocationID = RPT_Warehouse_Response.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
				WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.LocationID,0)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
				UPDATE	UnfilteredResponse
				SET		SupplierID = RPT_Warehouse_Response.SupplierID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
				WHERE ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(RPT_Warehouse_Response.SupplierID,0)
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
				UPDATE	UnfilteredResponse
				SET		CustomerID = PM_VW_StockCode.CustomerID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID INNER JOIN
							PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID
				WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

			IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
				UPDATE	UnfilteredResponse
				SET		StockCodeID = RPT_Warehouse_Response.StockCodeID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
		END
	ELSE --don't use warehouse table
		BEGIN
			DECLARE @override bit 
			SET @override = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
			
			INSERT INTO @UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
			SELECT ResponseID, ResponseStatus,
					--check config value to use Actual or Override QAFlagID
					CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
			FROM	QMS_Response
			WHERE	(RecordStatus < @RecordStatus) AND (EndDate BETWEEN @DateFrom AND @DateTo)
			
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
				UPDATE	UnfilteredResponse
				SET		LocationID = Location.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
				ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
				UPDATE	UnfilteredResponse
				SET		LocationID = Location.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
				ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
				UPDATE	UnfilteredResponse
				SET		SupplierID = QMS_ResponseHeader.Header_SourceID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
				ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
				UPDATE	UnfilteredResponse
				SET		CustomerID = PM_VW_StockCode.CustomerID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID INNER JOIN
							PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
				WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

			IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
				UPDATE	UnfilteredResponse
				SET		StockCodeID = HeaderStockCode.StockCodeID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID
				where ISNULL(UnfilteredResponse.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)
		END
	END

	-- Only return responses which match the filter specified, or for which we do not have a filter.
	-- This means that admin users (any user with no permissions set) can see all responses.
	INSERT INTO @Response (ResponseID, ResponseStatus, QAFlagID)
	SELECT	ResponseID, ResponseStatus, QAFlagID
	FROM	@UnfilteredResponse UnfilteredResponse
	WHERE (	NOT EXISTS (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL)
		OR	LocationID IN (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
		OR	SupplierID IN (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL)
		OR	CustomerID IN (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL) )
	AND (	StockCodeID IS NULL -- Allow unlinked stockcode responses
		OR	NOT EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		OR	StockCodeID IN (SELECT StockCodeID FROM @Table_StockCode) )
	
	RETURN
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_AssessmentType_IsQAS')
	EXEC('CREATE FUNCTION [dbo].[FNC_AssessmentType_IsQAS]() RETURNS bit AS BEGIN RETURN 0 END')
GO
--=============================================
-- Author:		MBS\Phil.Harding
-- Create date: 09/03/2015
-- Version:		1
-- Description: Indicates whether an AssessmentType is a QAS AssessmentType (refering to dbo.QAS_AssessmentType)
--=============================================
-- PH TFS 13-16159: First iteration (Helps hide/show "Edit Check" icon)
--=============================================
ALTER FUNCTION [dbo].[FNC_AssessmentType_IsQAS] ( @AssessmentTypeID int )
RETURNS bit

AS
BEGIN

DECLARE @result bit = 0
DECLARE @recordCount int

SELECT @recordCount = COUNT(*)
FROM [ef].[VW_QAS_AssessmentType]
WHERE AssessmentTypeID = @AssessmentTypeID

IF (@recordCount > 0) BEGIN SET @result = 1 END

RETURN @result

END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_QuestionPool_IsSampleCount')
	EXEC('CREATE FUNCTION [dbo].[FNC_QuestionPool_IsSampleCount]() RETURNS bit AS BEGIN RETURN 0 END')
GO
--=============================================
-- Author:		MBS\Phil.Harding
-- Create date: 09/03/2015
-- Version:		1
-- Description: Indicates whether a Question is a Sample Count
--=============================================
-- PH TFS 13-16159: First iteration
--=============================================
ALTER FUNCTION [dbo].[FNC_QuestionPool_IsSampleCount] ( @QuestionID int )
RETURNS bit

AS
BEGIN

DECLARE @result bit = 0
DECLARE @recordCount int

SELECT @recordCount = COUNT(*)
FROM [dbo].[QMS_QuestionPool]
WHERE QuestionID = @QuestionID
AND Sys_Question = 'SampleCount'

IF (@recordCount > 0) BEGIN SET @result = 1 END

RETURN @result

END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_QuestionPool_IsSampleWeight')
	EXEC('CREATE FUNCTION [dbo].[FNC_QuestionPool_IsSampleWeight]() RETURNS bit AS BEGIN RETURN 0 END')
GO
--=============================================
-- Author:		MBS\Phil.Harding
-- Create date: 09/03/2015
-- Version:		1
-- Description: Indicates whether a Question is a Sample Weight
--=============================================
-- PH TFS 13-16159: First iteration
--=============================================
ALTER FUNCTION [dbo].[FNC_QuestionPool_IsSampleWeight] ( @QuestionID int )
RETURNS bit

AS
BEGIN

DECLARE @result bit = 0
DECLARE @recordCount int

SELECT @recordCount = COUNT(*)
FROM [dbo].[QMS_QuestionPool]
WHERE QuestionID = @QuestionID
AND Sys_Question = 'SampleWeight'

IF (@recordCount > 0) BEGIN SET @result = 1 END

RETURN @result

END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_GENERATE_CorrectiveAction_UserList')
	EXEC('CREATE FUNCTION [dbo].[FNC_GENERATE_CorrectiveAction_UserList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Jon Edwards
-- Create date: 06/01/2015
-- Description:	Returns list of users with visibility of a Corrective Action
-- Version: 5
-- =============================================
-- JE TFS 13-9973: created
-- JE TFS 13-14013: corrected returning of users with no Location permissions
-- JT TFS 13-16507: adding admin locationpermissions no longer return users already added to @Users table.
-- JT TFS 13-16865: Re-do logic for permissions.
-- =============================================
ALTER FUNCTION dbo.FNC_GENERATE_CorrectiveAction_UserList
(
	@CorrectiveActionID int
)
RETURNS
@Users TABLE
(
	UserID int
)
AS
BEGIN

	--get CA's LocationID, SupplierID and whether it's an ad-hoc ca.
	DECLARE @LocationID int
	DECLARE @SupplierID int
	DECLARE @IsCAAdHoc bit
	DECLARE @IsRaisedFromCA bit

	SELECT	@LocationID = LocationID, 
			@SupplierID = AssignSupplierID,
			@IsCAAdHoc = CASE WHEN CASourceID = 4 THEN 1 ELSE 0 END,
			@IsRaisedFromCA = CASE WHEN CASourceID = 3 THEN 1 ELSE 0 END
	FROM QMS_CorrectiveAction WHERE CorrectiveActionID = @CorrectiveActionID

	--get StockCodeID for CA linked at check level
	DECLARE @StockCodeID int
	SELECT	@StockCodeID = QMS_ResponseHeader.Header_SourceID
	FROM	QMS_ResponseHeader
			INNER JOIN QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			INNER JOIN QMS_CorrectiveAction ON QMS_ResponseHeader.ResponseID = QMS_CorrectiveAction.SourceObjectID
	WHERE	(QMS_CorrectiveAction.CorrectiveActionID = @CorrectiveActionID)
			AND (QMS_CorrectiveAction.CASourceID = 1)
			AND (QMS_AssessmentHeader.DataTypeID = 9)
			AND (QMS_AssessmentHeader.DataLookupID = 2)

	--get StockCodeID for CA linked at question level if CA not linked at check level
	IF @StockCodeID IS NULL
		SELECT	@StockCodeID = QMS_ResponseHeader.Header_SourceID
		FROM	QMS_ResponseQuestion
				INNER JOIN QMS_ResponseHeader ON QMS_ResponseQuestion.ResponseID = QMS_ResponseHeader.ResponseID
				INNER JOIN QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID		
				INNER JOIN QMS_CorrectiveAction ON QMS_ResponseQuestion.ResponseQuestionID = QMS_CorrectiveAction.SourceObjectID
		WHERE	(QMS_CorrectiveAction.CorrectiveActionID = @CorrectiveActionID)
				AND (QMS_CorrectiveAction.CASourceID = 2)
				AND (QMS_AssessmentHeader.DataTypeID = 9)
				AND (QMS_AssessmentHeader.DataLookupID = 2)

	--get BusinessGroup and StockCodeSubGroup for CA's StockCodeID (if CA has one)
	IF @StockCodeID IS NOT NULL
	BEGIN
		DECLARE @BusinessGroup AS nvarchar(50)
		SELECT @BusinessGroup = BusinessGroup FROM PM_VW_StockCode WHERE StockCodeID = @StockCodeID

		DECLARE @StockCodeSubGroup AS nvarchar(50)
		SELECT @StockCodeSubGroup = StockCodeSubGroup FROM PM_VW_StockCode WHERE StockCodeID = @StockCodeID
	END

	DECLARE @tmpUsers TABLE (
		UserID int PRIMARY KEY,
		IsAdminUser bit,

		IsSupplier bit,
		HasSupplierSitePermissions bit DEFAULT(0),
		CanSeeWithSitePermissions bit DEFAULT(0),
		CanSeeWithSupplierPermissions bit DEFAULT(0),

		HasLocationPermissions bit DEFAULT(0),
		CanSeeWithLocationPermissions bit DEFAULT(0),
		HasStockCodePermissions bit DEFAULT(0),
		CanSeeWithStockCodePermissions bit DEFAULT(0),
		CAHasNoLocation bit,
		IsCAAdHoc bit,
		IsRaisedFromCA bit
	)

	INSERT @tmpUsers (UserID, IsAdminUser, IsSupplier, CAHasNoLocation, IsCAAdHoc, IsRaisedFromCA)
	SELECT UserID, 
			CASE WHEN levelOrder = 1 THEN 1 ELSE 0 END as IsAdminUser,
			CASE WHEN levelOrder = 9 THEN 1 ELSE 0 END AS IsSupplier,
			CASE WHEN ISNULL(@LocationID, 0) = 0 THEN 1 ELSE 0 END,
			@IsCAAdHoc,
			@IsRaisedFromCA
	FROM (
		SELECT UserID, 
			userLevel as levelOrder -- to eliminate it as a cause - should be changed in other SP use line below instead when releasing
			--dbo.FNC_GetUserOrderLevel(UserID) levelOrder
		FROM QMS_User
		WHERE RecordStatus < 201
	) as levelOrders


	
	UPDATE users
		SET HasSupplierSitePermissions = user_permissions.HasLocationPermissions,
			HasLocationPermissions = user_permissions.HasLocationPermissions,
			HasStockCodePermissions = user_permissions.HasStockCodePermissions
	FROM @tmpUsers users
	INNER JOIN (
		SELECT UserID, 
				max(CASE WHEN PermissionType = 103 THEN 1 ELSE 0 END) HasLocationPermissions,
				max(CASE WHEN PermissionType = 101 THEN 1 ELSE 0 END) HasSupplierPermissions,
				max(CASE WHEN PermissionType IN (104, 105, 106) THEN 1 ELSE 0 END) HasStockCodePermissions
		FROM QMS_UserPermissions 
		WHERE PermissionAllow = 1
		GROUP BY UserID
	) as user_permissions ON users.UserID = user_permissions.UserID

	
	-- set users who have location permissions to see this CA
	UPDATE users
		SET CanSeeWithLocationPermissions = 1, CanSeeWithSitePermissions = 1
	FROM	QMS_User
	INNER JOIN @tmpUsers users ON QMS_User.UserID = users.UserID 
	INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	(PermissionType = 103) AND (PermissionAllow = 1) AND (PermissionTitle = @LocationID)
	AND QMS_User.RecordStatus < 201


	--get users with permissions to the SupplierID
	UPDATE users
		SET CanSeeWithSupplierPermissions = 1
	FROM	QMS_User
	INNER JOIN @tmpUsers users ON QMS_User.UserID = users.UserID 
	INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	(PermissionType = 101) AND (PermissionAllow = 1) AND (PermissionTitle = @SupplierID)
	AND QMS_User.RecordStatus < 201

	--get users with permissions to the StockCodeID (need to check three levels of hierarchy: Business Group, Sub Group and Stock Code)
	UPDATE users
		SET CanSeeWithStockCodePermissions = 1
	FROM	QMS_User
	INNER JOIN @tmpUsers users ON QMS_User.UserID = users.UserID 
	INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	((PermissionType = 106) AND (PermissionAllow = 1) AND (PermissionTitle = @BusinessGroup))
	AND QMS_User.RecordStatus < 201

	UPDATE users
		SET CanSeeWithStockCodePermissions = 1
	FROM	QMS_User
	INNER JOIN @tmpUsers users ON QMS_User.UserID = users.UserID 
	INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	((PermissionType = 105) AND (PermissionAllow = 1) AND (PermissionTitle = @StockCodeSubGroup))
	AND QMS_User.RecordStatus < 201

	UPDATE users
		SET CanSeeWithStockCodePermissions = 1
	FROM	QMS_User
	INNER JOIN @tmpUsers users ON QMS_User.UserID = users.UserID 
	INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	((PermissionType = 104) AND (PermissionAllow = 1) AND (PermissionTitle = @StockCodeID))
	AND QMS_User.RecordStatus < 201

	-- NOTE: IsAdminUser only overrides location permissions, not stockcode permissions.

	INSERT @Users
	SELECT UserID FROM @tmpUsers
	WHERE IsRaisedFromCA = 0
	AND (
			(
				IsSupplier = 1 
				AND (
						(HasSupplierSitePermissions = 1 AND CanSeeWithSitePermissions = 1)
						OR
						(HasSupplierSitePermissions = 0 AND CanSeeWithSupplierPermissions = 1)
				)
			)
			OR
			(	IsSupplier = 0 
				AND (
					(HasLocationPermissions = 1 AND HasStockCodePermissions = 1 AND (IsAdminUser = 1 OR CAHasNoLocation = 1 OR CanSeeWithLocationPermissions = 1) AND (CanSeeWithStockCodePermissions = 1 OR IsCAAdHoc = 1))
					OR
					(HasLocationPermissions = 1 AND HasStockCodePermissions = 0 AND (IsAdminUser = 1 OR CAHasNoLocation = 1 OR CanSeeWithLocationPermissions = 1))
					OR 
					(HasLocationPermissions = 0 AND HasStockCodePermissions = 1 AND (CanSeeWithStockCodePermissions = 1 OR IsCAAdHoc = 1))
					OR
					(HasLocationPermissions = 0 AND HasStockCodePermissions = 0)
				)
			)
	)

	RETURN
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_GENERATE_User_CorrectiveActionList')
	EXEC('CREATE FUNCTION [dbo].[FNC_GENERATE_User_CorrectiveActionList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Jon Edwards
-- Create date: 27/01/2012
-- Description:	Returns list of Corrective Actions
-- Version: 2
-- =============================================
-- JE TFS 13-9973: created
-- JT TFS 13-16683: Added Primary key to @Table_LocationCorrectiveAction to speed up slowest part of query.
-- =============================================
ALTER FUNCTION dbo.FNC_GENERATE_User_CorrectiveActionList 
(
	@UserID int,
	@IsActive bit
)
RETURNS
@CorrectiveActions TABLE
(
	CorrectiveActionID int
)
AS
BEGIN

	--Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserLevel = 1 --Admin can see hidden/void checks
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 180
	
	DECLARE @Table_Location TABLE (LocationID int)
	
	--updated in TFS 2757 - only make CAs available to Suppliers if CA is associated with permitted supplier
	IF @UserLevel = 9
	BEGIN
	
		--Get table of available locations
		INSERT INTO @Table_Location (LocationID)
		SELECT     CAST(PermissionTitle AS int) AS LocationID
		FROM         QMS_UserPermissions
		WHERE     (PermissionType = 103) AND (PermissionAllow = 1) AND (UserID = @UserID)
		
		--Get table of permitted Suppliers for user
		DECLARE @Table_Supplier TABLE (SupplierID int)
		INSERT INTO @Table_Supplier (SupplierID)
		SELECT		PermissionTitle AS SupplierID
		FROM		QMS_UserPermissions
		WHERE		(PermissionType = 101) AND (PermissionAllow = 1) AND (UserID = @UserID)		
	
		--If just Supplier permissions
		IF (SELECT COUNT (LocationID) FROM @Table_Location)=0 AND (SELECT COUNT (SupplierID) FROM @Table_Supplier)>0
		--Check CA status
		BEGIN
			--get CAs which are assigned to Supplier(s) user has permission on
			INSERT INTO @CorrectiveActions(CorrectiveActionID)
			SELECT	QMS_CorrectiveAction.CorrectiveActionID
			FROM	QMS_CorrectiveAction INNER JOIN
						@Table_Supplier AS Supplier ON QMS_CorrectiveAction.AssignSupplierID = Supplier.SupplierID
			WHERE	((@IsActive = 1) AND (CAStatusID IN (1,2,3,6)))
					OR
					((@IsActive = 0) AND (CAStatusID NOT IN (1,2,3,6)))
		END
		
		--If Supplier AND Location permissions
		IF (SELECT COUNT (LocationID) FROM @Table_Location)>0 AND (SELECT COUNT (SupplierID) FROM @Table_Supplier)>0
		BEGIN
			--Table of ResponseIDs linked to locations
			DECLARE @Table_ResponseLocationSupplier TABLE (ResponseID int, ResponseStatus int, LocationID int)
			INSERT INTO @Table_ResponseLocationSupplier (ResponseID, ResponseStatus, LocationID)
			SELECT	QMS_Response.ResponseID, QMS_Response.ResponseStatus, User_Location.LocationID
			FROM	QMS_Response INNER JOIN
						  (SELECT	ResponseHeader_Location.ResponseID, ResponseHeader_Location.Header_SourceID AS LocationID
							FROM		QMS_ResponseHeader AS ResponseHeader_Location INNER JOIN
											QMS_AssessmentHeader AS AssessmentHeader_Location ON 
											ResponseHeader_Location.AssessmentHeaderID = AssessmentHeader_Location.AssessmentHeaderID INNER JOIN
											@Table_Location AS Location ON ResponseHeader_Location.Header_SourceID = Location.LocationID
							WHERE	(AssessmentHeader_Location.DataTypeID = 9) AND (AssessmentHeader_Location.DataLookupID = 4 OR 
											AssessmentHeader_Location.DataLookupID = 53)) AS User_Location ON 
					QMS_Response.ResponseID = User_Location.ResponseID
			WHERE     (QMS_Response.RecordStatus < @RecordStatus)

			--get CAs linked to Suppliers and Locations user has permissions on - where CA linked to question
			INSERT INTO @CorrectiveActions(CorrectiveActionID)
			SELECT	CorrectiveActionID
			FROM	QMS_CorrectiveAction INNER JOIN
						@Table_Supplier AS Supplier ON QMS_CorrectiveAction.AssignSupplierID = Supplier.SupplierID LEFT OUTER JOIN
						QMS_ResponseQuestion ON QMS_CorrectiveAction.SourceObjectID = QMS_ResponseQuestion.ResponseQuestionID LEFT OUTER JOIN
						@Table_ResponseLocationSupplier AS ResponseLocationSupplier ON QMS_ResponseQuestion.ResponseID = ResponseLocationSupplier.ResponseID
			WHERE	((@IsActive = 1 AND CAStatusID IN (1,2,3,6) AND CASourceID = 2) --Check for CAs that are New, In Progress, Failed or Rejected
					OR
					(@IsActive = 0 AND CAStatusID NOT IN (1,2,3,6))  AND CASourceID = 2)
					--suppliers with supplier permissions will always have location permissions for that supplier too, so check responses have a location supplier has permission for
					AND ResponseLocationSupplier.LocationID IS NOT NULL
					
			--get CAs linked to Suppliers and Locations user has permissions on - where CA linked to check
			INSERT INTO @CorrectiveActions(CorrectiveActionID)
			SELECT	CorrectiveActionID
			FROM	QMS_CorrectiveAction INNER JOIN
						@Table_Supplier AS Supplier ON QMS_CorrectiveAction.AssignSupplierID = Supplier.SupplierID LEFT OUTER JOIN
						@Table_ResponseLocationSupplier AS ResponseLocationSupplier ON QMS_CorrectiveAction.SourceObjectID = ResponseLocationSupplier.ResponseID
			WHERE	((@IsActive = 1 AND CAStatusID IN (1,2,3,6) AND CASourceID = 1) --Check for CAs that are New, In Progress, Failed or Rejected
					OR
					(@IsActive = 0 AND CAStatusID NOT IN (1,2,3,6))  AND CASourceID = 1)
					--suppliers with supplier permissions will always have location permissions for that supplier too, so check responses have a location supplier has permission for
					AND ResponseLocationSupplier.LocationID IS NOT NULL
		END
	END
	
--User Level is not 9 (Supplier), check Stock Code hierarchy permissions if they are assigned and that user's Location permissions
	IF @UserLevel <> 9
	BEGIN
		--Get table of available Locations user has permissions for
		INSERT INTO @Table_Location (LocationID)
		SELECT	CAST(PermissionTitle AS int) AS LocationID
		FROM	QMS_UserPermissions
		WHERE	(PermissionType = 103) AND (PermissionAllow = 1) AND (UserID = @UserID)

		--get table of CAs that are linked to above Locations
		DECLARE @Table_LocationCorrectiveAction TABLE (
			CorrectiveActionID int PRIMARY KEY, 
			SourceObjectID int, 
			CAStatusID int, 
			CASourceID int
			UNIQUE (SourceObjectID, CorrectiveActionID)
		)
		INSERT INTO @Table_LocationCorrectiveAction (CorrectiveActionID, SourceObjectID, CAStatusID, CASourceID)
		SELECT	CorrectiveActionID, SourceObjectID, CAStatusID, CASourceID
		FROM	QMS_CorrectiveAction
		WHERE	((@UserLevel <> 1) AND (LocationID IN (SELECT LocationID FROM @Table_Location))) --restrict non-admin users visiblity of CAs to those assigned to their Location(s) only
					OR
				(@UserLevel = 1) --Admin can access all CAs
					OR
				((@UserLevel <> 1) AND (SELECT COUNT(LocationID) FROM @Table_Location) = 0) --show all CAs if user has NO Location permissions
						OR
				((QMS_CorrectiveAction.LocationID IS NULL) OR (QMS_CorrectiveAction.LocationID = 0)) --also include all CAs that aren't assigned to a Location
	
		--get Stock Code permissions for user
		DECLARE @Table_UserStockCodes TABLE (ResponseID int, StockCodeID int)
		INSERT INTO @Table_UserStockCodes (ResponseID, StockCodeID)
		SELECT	ResponseHeader_StockCode.ResponseID, ResponseHeader_StockCode.Header_SourceID AS StockCodeID
		FROM	QMS_ResponseHeader AS ResponseHeader_StockCode INNER JOIN
					QMS_AssessmentHeader AS AssessmentHeader_StockCode ON ResponseHeader_StockCode.AssessmentHeaderID = AssessmentHeader_StockCode.AssessmentHeaderID INNER JOIN
					dbo.FNC_User_StockCodeList(@UserID) AS StockCodes ON ResponseHeader_StockCode.Header_SourceID = StockCodes.StockCodeID
		WHERE	(AssessmentHeader_StockCode.DataTypeID = 9) AND (AssessmentHeader_StockCode.DataLookupID = 2)
		
		DECLARE @ca_staging TABLE (CorrectiveActionID int PRIMARY KEY)
		IF  (SELECT COUNT (UserPermissionID) FROM QMS_UserPermissions WHERE (UserID = @UserID) AND (PermissionAllow = 1) AND (PermissionType IN (104,105,106))) > 0 --user has Stock Code permissions, so check these
		BEGIN
			--get Question Level CAs linked to StockCodes user has permission to
			INSERT @ca_staging (CorrectiveActionID)
			SELECT	LocationCorrectiveAction.CorrectiveActionID
			FROM	@Table_LocationCorrectiveAction AS LocationCorrectiveAction INNER JOIN
						QMS_ResponseQuestion ON LocationCorrectiveAction.SourceObjectID = QMS_ResponseQuestion.ResponseQuestionID INNER JOIN
						@Table_UserStockCodes AS User_StockCodes ON	QMS_ResponseQuestion.ResponseID = User_StockCodes.ResponseID
			WHERE	((@IsActive = 1) AND (CAStatusID IN (1,2,3,6)) AND (CASourceID = 2))
					OR
					((@IsActive = 0) AND (CAStatusID NOT IN (1,2,3,6)) AND (CASourceID = 2))
		
			--get CAs linked at check level
			INSERT @ca_staging (CorrectiveActionID)
			SELECT	LocationCorrectiveAction.CorrectiveActionID
			FROM	@Table_LocationCorrectiveAction AS LocationCorrectiveAction INNER JOIN
						@Table_UserStockCodes AS User_StockCodes ON	LocationCorrectiveAction.SourceObjectID = User_StockCodes.ResponseID
			WHERE	((@IsActive = 1) AND (CAStatusID IN (1,2,3,6)) AND (CASourceID = 1))
					OR
					((@IsActive = 0) AND (CAStatusID NOT IN (1,2,3,6)) AND (CASourceID = 1))
			
			--get Ad-Hoc CAs (not linked to a Response)
			INSERT @ca_staging (CorrectiveActionID)
			SELECT	LocationCorrectiveAction.CorrectiveActionID
			FROM	@Table_LocationCorrectiveAction AS LocationCorrectiveAction
			WHERE	(((@IsActive = 1) AND (CASourceID = 4) AND (CAStatusID IN (1,2,3,6)))
					OR
					((@IsActive = 0) AND (CASourceID = 4) AND (CAStatusID NOT IN (1,2,3,6))))
		END
		ELSE
		BEGIN
			--get Question Level CAs
			INSERT INTO @ca_staging(CorrectiveActionID)
			SELECT	LocationCorrectiveAction.CorrectiveActionID
			FROM	@Table_LocationCorrectiveAction AS LocationCorrectiveAction 
			--INNER JOIN	QMS_ResponseQuestion ON LocationCorrectiveAction.SourceObjectID = QMS_ResponseQuestion.ResponseQuestionID
			WHERE	((@IsActive = 1) AND (CAStatusID IN (1,2,3,6)) AND (CASourceID = 2))
					OR
					((@IsActive = 0) AND (CAStatusID NOT IN (1,2,3,6)) AND (CASourceID = 2))

					
			--get CAs linked at check level and Ad-Hoc CAs (not linked to a Response)
			INSERT INTO @ca_staging(CorrectiveActionID)
			SELECT	LocationCorrectiveAction.CorrectiveActionID
			FROM	@Table_LocationCorrectiveAction AS LocationCorrectiveAction
			WHERE	((@IsActive = 1) AND (CAStatusID IN (1,2,3,6)) AND (CASourceID IN (1,4)))
					OR
					((@IsActive = 0) AND (CAStatusID NOT IN (1,2,3,6)) AND (CASourceID IN (1,4)))
		END

		INSERT INTO @CorrectiveActions(CorrectiveActionID)
		SELECT CorrectiveActionID FROM @ca_staging 
		GROUP BY CorrectiveActionID

	END
	RETURN
END
GO


--PRINT 'Updating Stored Procedures...'
--GO
IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CountryLocales_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CountryLocales_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 10/08/2010
-- Description:	Gets data to display in country locale combo
-- Version:		2
-- =============================================
-- AC TFS 1200 - Only gets locales that exist in the APP_ResourceLocale table
-- =============================================
ALTER PROCEDURE QRY_CountryLocales_Combo
AS
BEGIN

	SELECT  APP_CountryLocales.CountryLocaleID, APP_CountryLocales.[Description]
	FROM         APP_CountryLocales 
	INNER JOIN APP_ResourceLocale ON APP_CountryLocales.CountryLocaleID = APP_ResourceLocale.CountryLocaleID
	GROUP BY APP_CountryLocales.CountryLocaleID, APP_CountryLocales.[Description]
	ORDER BY Description

END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CustomResource_List')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CustomResource_List] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 07/01/2011
-- Description:	Gets custom resources for user
-- version: 2
-- =============================================
-- jf tfs 1202 - Change locale culture based on login user
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CustomResource_List]
	@UserID int,
	@CountryLocaleCode nvarchar(255) = NULL
AS

DECLARE @CountryLocaleID int
--IF @CountryLocaleCode IS NOT NULL
--	SELECT     @CountryLocaleID = CountryLocaleID
--	FROM         APP_CountryLocales
--	WHERE     (StringCode = @CountryLocaleCode)
--ELSE
	SELECT     @CountryLocaleID = ISNULL(LocaleID, 46)
	FROM         QMS_User
	WHERE     (UserID = @UserID)

SELECT     APP_Resource.ResourceName, QMS_CustomResource.CountryLocaleID, QMS_CustomResource.ResourceString
FROM         QMS_CustomResource INNER JOIN
                      APP_Resource ON QMS_CustomResource.ResourceID = APP_Resource.ResouceID
WHERE     (QMS_CustomResource.RecordStatus < 201) AND (QMS_CustomResource.CountryLocaleID = @CountryLocaleID OR
                      QMS_CustomResource.CountryLocaleID IS NULL)
ORDER BY APP_Resource.ResourceName, QMS_CustomResource.CountryLocaleID DESC
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_PriorityLevel_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_PriorityLevel_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 09/07/2008
-- Description:	Gets list of priority levels to fill combo
-- Version: 2
-- =============================================
-- jf tfs 1202 - get test from language strings
-- =============================================
ALTER PROCEDURE [dbo].[QRY_PriorityLevel_Combo] 
	@UserID as int
AS

SELECT     PriorityLevelID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), PriorityLevel) as PriorityLevel
FROM         APP_PriorityLevel
ORDER BY ListIndex


GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_QAAction_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_QAAction_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 18/11/2008
-- Description:	Gets list of QA Actions to fill combo
-- version: 2
-- =============================================
-- jf tfs 1202 - get the string from language table
-- =============================================
ALTER PROCEDURE [dbo].[QRY_QAAction_Combo] 
	@QAFlagID int = 0,
	@UserID int = 0
AS


IF (SELECT COUNT(QAFlagActionID) FROM APP_QAFlagAction WHERE (QAFlagID = @QAFlagID AND Hide = 0))=0  --no filter for QA flag
	SELECT     QAActionID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAAction) + CASE WHEN ActionCode IS NOT NULL THEN ' (' + ActionCode + ')' ELSE '' END AS ActionText
	FROM         APP_QAAction
	WHERE     (UserEnabled = 1)
	ORDER BY ListIndex
	
ELSE
	SELECT     APP_QAAction.QAActionID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), APP_QAAction.QAAction) + CASE WHEN ActionCode IS NOT NULL THEN ' (' + ActionCode + ')' ELSE '' END AS ActionText
	FROM         APP_QAAction INNER JOIN
						  APP_QAFlagAction ON APP_QAAction.QAActionID = APP_QAFlagAction.QAActionID
	WHERE     (APP_QAAction.UserEnabled = 1) AND (APP_QAFlagAction.QAFlagID = @QAFlagID) AND (APP_QAFlagAction.Hide = 0)
	ORDER BY APP_QAAction.ListIndex
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_ResourceLocale_List')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_ResourceLocale_List] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 07/01/2011
-- Description:	Gets resources for current locale
-- version: 2
-- =============================================
-- jf tfs 1202 - Change locale culture based on login user
-- =============================================
ALTER PROCEDURE [dbo].[QRY_ResourceLocale_List]
	@UserID int,
	@CountryLocaleCode nvarchar(255) = NULL
AS

DECLARE @CountryLocaleID int
--IF @CountryLocaleCode IS NOT NULL
--	SELECT     @CountryLocaleID = CountryLocaleID
--	FROM         APP_CountryLocales
--	WHERE     (StringCode = @CountryLocaleCode)
--ELSE
	SELECT     @CountryLocaleID = ISNULL(LocaleID, 46)
	FROM         QMS_User
	WHERE     (UserID = @UserID)

SELECT     APP_Resource.ResourceName, APP_ResourceLocale.ResourceString
FROM         APP_Resource INNER JOIN
                      APP_ResourceLocale ON APP_Resource.ResouceID = APP_ResourceLocale.ResourceID
WHERE     (APP_ResourceLocale.CountryLocaleID = @CountryLocaleID)
ORDER BY APP_Resource.ResourceName
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_TaskView_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_TaskView_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 20/08/2008
-- Description:	Get data to fill combo in task view
-- Version: 2
-- =============================================
-- jf tfs 1202 - get text from language strings
-- =============================================
ALTER PROCEDURE [dbo].[QRY_TaskView_Combo] 
	@UserID as int
AS

SELECT     TaskViewID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), TaskView) as TaskView
FROM         APP_TaskView
ORDER BY ListIndex
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_User_AdminList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_User_AdminList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 27/06/2008
-- Description:	Gets list of users for admin
-- Version:		7
-- =============================================
-- JE TFS 2677 added new UserLevel logic
-- MP 18/09/12 change to selection of avalable user level as not sure why QMS_User was included here
-- JT TFS 6623 added IsUserLockedOut column (based on logic in clsUserAdmin.IsUserLockedOut, and added SearchUserID
-- JT TFS 7115 Added begin/end and NOCOUNT options to see if they fixed slowdown.
-- JT TFS 8022 - Return GUID so that it can be passed to LogiXML
-- JT TFS 13-11810 - Include CustomerID in output.
-- AC TFS 1200 - Include Lookup_LocaleID in output
-- =============================================
ALTER PROCEDURE dbo.QRY_User_AdminList 
	@UserID int,
	@Company nvarchar(255) = NULL,
	@SearchUserID int = NULL
AS
BEGIN

	SET NOCOUNT ON 
	-- temp table to hold user levels
	DECLARE @AvailableUserLevels TABLE
	(
		UserLevelID int,
		UserLevel nvarchar(50),
		LevelOrder int
	)
	-- user level of selected User
	DECLARE @LevelOrder int
	-- match UserID to get LevelOrder
	SET @LevelOrder = (SELECT LevelOrder FROM APP_UserLevel INNER JOIN
							QMS_User ON APP_UserLevel.UserLevelID = QMS_User.UserLevel
						WHERE UserID = @UserID)

	IF @LevelOrder = 1
		-- insert available (i.e. lower) user levels into temp table then join on this in main query
		INSERT INTO @AvailableUserLevels (UserLevelID, UserLevel, LevelOrder)
		SELECT     UserLevelID, UserLevel, LevelOrder
		FROM         APP_UserLevel
		WHERE     (Hide = 0) AND (LevelOrder >= @LevelOrder)
	ELSE
		-- insert available (i.e. lower) user levels into temp table then join on this in main query
		INSERT INTO @AvailableUserLevels (UserLevelID, UserLevel, LevelOrder)
		SELECT     UserLevelID, UserLevel, LevelOrder
		FROM         APP_UserLevel
		WHERE     (Hide = 0) AND (LevelOrder > @LevelOrder)

	SET NOCOUNT OFF

	SELECT DISTINCT
		QMS_User.UserID, 
		QMS_User.FullName, 
		QMS_User.UserName, 
		QMS_User.Password, 
		QMS_User.Position, 
		QMS_User.Company, 
		QMS_User.EMail, 
		QMS_User.Mobile, 
		QMS_User.PIN, 
		QMS_User.LeftCompany, 
		QMS_User.UserLevel, 
		AvailableUserLevels.UserLevel AS Lookup_UserLevel, 
		QMS_User.LocaleID, 
		APP_CountryLocales.[Description] AS Lookup_LocaleID, 
		QMS_User.IsUserGroup, 
		QMS_User.ParentUserID, 
		Lookup_UserGroup.FullName AS Lookup_ParentUserID, 
		CAST(CASE	WHEN ISNULL(SupplierPermissions.NumPermissions, 0) > 1 THEN 0 
					WHEN ISNULL(SupplierPermissions.NumPermissions, 0) = 1 THEN SupplierID 
					ELSE NULL END AS int) AS SupplierID, 
		CAST(CASE	WHEN ISNULL(SupplierPermissions.NumPermissions, 0) > 1 THEN N'(Multiple)' 
					WHEN ISNULL(SupplierPermissions.NumPermissions, 0) = 1 THEN	ISNULL((	SELECT     SupplierName
																					FROM          PM_VW_Supplier
																					WHERE      (SupplierID = SupplierPermissions.SupplierID)), '(deleted supplier)')
					ELSE NULL END AS nvarchar) AS Lookup_SupplierID, 

		CAST(CASE	WHEN ISNULL(CustomerPermissions.NumPermissions, 0) > 1 THEN 0 
					WHEN ISNULL(CustomerPermissions.NumPermissions, 0) = 1 THEN CustomerID 
					ELSE NULL END AS int) AS CustomerID, 
		CAST(CASE	WHEN ISNULL(CustomerPermissions.NumPermissions, 0) > 1 THEN N'(Multiple)' 
					WHEN ISNULL(CustomerPermissions.NumPermissions, 0) = 1 THEN	ISNULL((	SELECT     CustomerName
																					FROM          PM_VW_Customer_Admin
																					WHERE      (CustomerID = CustomerPermissions.CustomerID)), '(deleted customer)') 
					ELSE NULL END AS nvarchar) AS Lookup_CustomerID, 
		CAST(OfficePermissions.NumPermissions AS nvarchar(100)) AS OfficeAccess, 
		CAST(MobilePermissions.NumPermissions AS nvarchar(100)) AS MobileAccess, 
		CAST(WebPermissions.NumPermissions AS nvarchar(100)) AS WebAccess, 
		CAST(SupplierPermissions.NumPermissions AS nvarchar(100)) AS SupplierAccess, 
		CAST(CustomerPermissions.NumPermissions AS nvarchar(100)) AS CustomerAccess, 
		CAST(LocationPermissions.NumPermissions AS nvarchar(100)) AS LocationAccess, 
		CAST(StockCodePermissions.NumPermissions AS nvarchar(100)) AS StockCodeAccess,
		CAST(CASE WHEN QMS_User.[Password] IS NULL THEN 1 ELSE 0 END as bit) Lookup_IsUserLockedOut,
		QMS_User.[GUID] AS Lookup_GUID
	FROM         QMS_User INNER JOIN
						   @AvailableUserLevels AS AvailableUserLevels ON QMS_User.UserLevel = AvailableUserLevels.UserLevelID LEFT OUTER JOIN
							  (SELECT     UserID, FullName
								FROM          QMS_User AS User_Group
								WHERE      (RecordStatus < 201) AND (IsUserGroup = 1)) AS Lookup_UserGroup ON QMS_User.ParentUserID = Lookup_UserGroup.UserID LEFT OUTER JOIN
							  (SELECT     UserID, COUNT(UserPermissionID) AS NumPermissions
								FROM          QMS_UserPermissions AS StockCode_Permissions
								WHERE      (PermissionAllow = 1) AND (PermissionType = 104)
								GROUP BY UserID) AS StockCodePermissions ON QMS_User.UserID = StockCodePermissions.UserID LEFT OUTER JOIN
							  (SELECT     UserID, COUNT(UserPermissionID) AS NumPermissions
								FROM          QMS_UserPermissions AS Location_Permissions
								WHERE      (PermissionAllow = 1) AND (PermissionType = 103)
								GROUP BY UserID) AS LocationPermissions ON QMS_User.UserID = LocationPermissions.UserID LEFT OUTER JOIN
							  (SELECT     UserID, COUNT(UserPermissionID) AS NumPermissions, MIN(CAST(PermissionTitle AS int)) AS SupplierID
								FROM          QMS_UserPermissions AS Supplier_Permissions
								WHERE      (PermissionAllow = 1) AND (PermissionType = 101)
								GROUP BY UserID) AS SupplierPermissions ON QMS_User.UserID = SupplierPermissions.UserID LEFT OUTER JOIN
							  (SELECT     UserID, COUNT(UserPermissionID) AS NumPermissions, MIN(CAST(PermissionTitle AS int)) AS CustomerID
								FROM          QMS_UserPermissions AS Customer_Permissions
								WHERE      (PermissionAllow = 1) AND (PermissionType = 102)
								GROUP BY UserID) AS CustomerPermissions ON QMS_User.UserID = CustomerPermissions.UserID LEFT OUTER JOIN
							  (SELECT     Web_Permissions.UserID, COUNT(Web_Permissions.UserPermissionID) AS NumPermissions
								FROM          QMS_UserPermissions AS Web_Permissions INNER JOIN
													   APP_Menu AS Web_Menu ON CAST(Web_Permissions.PermissionTitle AS int) = Web_Menu.MenuID
								WHERE      (Web_Permissions.PermissionAllow = 1) AND (Web_Permissions.PermissionType = 0) AND (Web_Menu.AppID = 3)
								GROUP BY Web_Permissions.UserID) AS WebPermissions ON QMS_User.UserID = WebPermissions.UserID LEFT OUTER JOIN
							  (SELECT     Mobile_Permissions.UserID, COUNT(Mobile_Permissions.UserPermissionID) AS NumPermissions
								FROM          QMS_UserPermissions AS Mobile_Permissions INNER JOIN
													   APP_Menu AS Mobile_Menu ON CAST(Mobile_Permissions.PermissionTitle AS int) = Mobile_Menu.MenuID
								WHERE      (Mobile_Permissions.PermissionAllow = 1) AND (Mobile_Permissions.PermissionType = 0) AND (Mobile_Menu.AppID = 2)
								GROUP BY Mobile_Permissions.UserID) AS MobilePermissions ON QMS_User.UserID = MobilePermissions.UserID LEFT OUTER JOIN
							  (SELECT     Office_Permissions.UserID, COUNT(Office_Permissions.UserPermissionID) AS NumPermissions
								FROM          QMS_UserPermissions AS Office_Permissions INNER JOIN
													   APP_Menu AS Office_Menu ON CAST(Office_Permissions.PermissionTitle AS int) = Office_Menu.MenuID
								WHERE      (Office_Permissions.PermissionAllow = 1) AND (Office_Permissions.PermissionType = 0) AND (Office_Menu.AppID = 1)
								GROUP BY Office_Permissions.UserID) AS OfficePermissions ON QMS_User.UserID = OfficePermissions.UserID LEFT JOIN APP_CountryLocales ON
								QMS_User.LocaleID = APP_CountryLocales.CountryLocaleID
	WHERE     (QMS_User.RecordStatus < 201) 
	AND (QMS_User.Company = @Company OR @Company IS NULL)
	AND (QMS_User.UserID = @SearchUserID OR @SearchUserID IS NULL)
	ORDER BY QMS_User.IsUserGroup, QMS_User.FullName
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_CorrectiveAction')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_CorrectiveAction] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 01/08/2008
-- Description:	Saves corrective action
-- Version:		12
-- =============================================
-- j.toolan 21 Feb 2012: Added ReminderDate
-- j.toolan  7 Mar 2012: Reset NotifcationSent on 6 if AssignSupplierID set (only if status changes)
-- MP TFS 5873 - Made sure SourceObjectID updated so that correct links to response created
-- JE TFS 7087 - added CorrectiveActionTypeID to UPDATE statement
-- JE TFS 7877 - corrected resetting of NotifcationSent for Completed alerts
-- JE TFS 7836 - updated to save LocationID CA is assigned
-- JT TFS 13-7950 - Make sure that if CorrectiveActionTypeID is specified, and CorrectiveActionEscalationID is NOT, then set it.
-- JE TFS 13-12635: call new SAVE_CorrectiveActionUser SP to save permissions data to QMS_CorrectiveActionUser table
-- MP TFS 13-15377: excluded call to new SAVE_CorrectiveActionUser SP for mobile
-- AC TFS 6335: Changes the escalation if the corrective action type or notification user group is changed
-- =============================================
ALTER PROCEDURE [dbo].[SAVE_CorrectiveAction]
	@CorrectiveActionID int,
	@CASourceID int,
	@SourceObjectID int,
	@CATitle nvarchar(50),
	@CASubTitle nvarchar(50),
	@CAReason nvarchar(1000),
	@CARequired nvarchar(1000),
	@CAStatusID int,
	@CorrectiveActionTypeID int,
	@CorrectiveActionEscalationID int,
	@NotificationGroupID int,
	@AssignUserID int,
	@DateOccurred datetime,
	@PriorityLevelID int,
	@DateLogged datetime,
	@DateDeadline datetime,
	@ActionTaken nvarchar(1000),
	@Comments nvarchar(1000),
	@DateCompleted datetime,
	@Completed_UserID int,
	@Completed_MachineName nvarchar(50),
	@DateSignedOff datetime,
	@SignOff_UserID int,
	@SignOff_MachineName nvarchar(50),
	@UserID int,
	@AssignSupplierID int,
	@ReminderDate datetime,
	@ReminderSent bit = NULL, -- This is not saved currently.
	@LocationID int
AS
BEGIN
	SET NOCOUNT ON

	-- BUG 7950 SP FIX
	SELECT @LocationID = NULLIF(@LocationID, 0) -- Do not allow zero for LocationID. It stops first new CA alert going out.

	DECLARE @DefaultCorrectiveActionEscalationID int
	SELECT @DefaultCorrectiveActionEscalationID = CorrectiveActionEscalationID 
	FROM QMS_CorrectiveActionEscalation 
	WHERE CorrectiveActionTypeID = @CorrectiveActionTypeID
	AND ListIndex = (	SELECT min(ListIndex) FROM QMS_CorrectiveActionEscalation 
						WHERE CorrectiveActionTypeID = @CorrectiveActionTypeID)
	-- END BUG 7950 fix

	IF ISNULL(@CorrectiveActionID,0)=0 --New record
	BEGIN
		INSERT INTO QMS_CorrectiveAction
						  (CASourceID, SourceObjectID, CATitle, CASubTitle, CAReason, CARequired, CAStatusID, CorrectiveActionTypeID, CorrectiveActionEscalationID, NotificationGroupID, 
						  AssignUserID, DateOccurred, PriorityLevelID, DateLogged, DateDeadline, Logged_UserID, Logged_MachineName, ActionTaken, Comments, DateCompleted, 
						  Completed_UserID, Completed_MachineName, AssignSupplierID, ReminderDate, LocationID)
		VALUES     (@CASourceID,@SourceObjectID,@CATitle,@CASubTitle,@CAReason,@CARequired,@CAStatusID,@CorrectiveActionTypeID,
					ISNULL(@CorrectiveActionEscalationID, @DefaultCorrectiveActionEscalationID),
					@NotificationGroupID,@AssignUserID,@DateOccurred,@PriorityLevelID,@DateLogged,@DateDeadline,@UserID,
						   HOST_NAME(),@ActionTaken,@Comments,@DateCompleted,@Completed_UserID,@Completed_MachineName,@AssignSupplierID, @ReminderDate, @LocationID)
	                      
		--Get new ID
		SELECT     @CorrectiveActionID = MAX(CorrectiveActionID)
		FROM         QMS_CorrectiveAction
	END
	ELSE --Update record
	BEGIN
		DECLARE @originalCAStatusID int
		SELECT	@originalCAStatusID = CAStatusID 
		FROM	QMS_CorrectiveAction
		WHERE	CorrectiveActionID = @CorrectiveActionID
		
		DECLARE @originalCorrectiveActionTypeID int
		DECLARE @originalNotificationGroupID int
		
		SELECT	@originalCorrectiveActionTypeID = CorrectiveActionTypeID, @originalNotificationGroupID = NotificationGroupID
		FROM	QMS_CorrectiveAction
		WHERE	CorrectiveActionID = @CorrectiveActionID
		
		-- Only update the escalation if the corrective action type or the notification user group has changed
		IF (@originalCorrectiveActionTypeID <> @CorrectiveActionTypeID OR @originalNotificationGroupID <> @NotificationGroupID)
		BEGIN
			SET @CorrectiveActionEscalationID = NULL
			SELECT @CorrectiveActionEscalationID = CorrectiveActionEscalationID
			FROM QMS_CorrectiveActionEscalation
			WHERE CorrectiveActionTypeId = @CorrectiveActionTypeId
			AND NotificationGroupId = @NotificationGroupId
		END

		UPDATE    QMS_CorrectiveAction
		SET              SourceObjectID = @SourceObjectID, CATitle = @CATitle, CASubTitle = @CASubTitle, CAReason = @CAReason, CARequired = @CARequired, 
							  CAStatusID = @CAStatusID, CorrectiveActionTypeID = @CorrectiveActionTypeID, NotificationGroupID = @NotificationGroupID, AssignUserID = @AssignUserID, DateOccurred = @DateOccurred, 
							  PriorityLevelID = @PriorityLevelID, DateDeadline = @DateDeadline, ActionTaken = @ActionTaken, Comments = @Comments, DateCompleted = @DateCompleted, 
							  Completed_UserID = @Completed_UserID, Completed_MachineName = @Completed_MachineName, DateSignedOff = @DateSignedOff, 
							  SignOff_UserID = @SignOff_UserID, SignOff_MachineName = @SignOff_MachineName, 
							  RecordStatus = CASE WHEN RecordStatus = 30 THEN 20 ELSE RecordStatus END, Updated = GETUTCDATE(), AssignSupplierID = @AssignSupplierID, 
							  ReminderDate = @ReminderDate, LocationID = @LocationID, CorrectiveActionEscalationID = @CorrectiveActionEscalationID
		WHERE     (CorrectiveActionID = @CorrectiveActionID)

		-- Only update notification flag if the status has changed.
		IF (@originalCAStatusID <> @CAStatusID)
		BEGIN
				--get Failure_NotificationGroupID for current CA escalation
			DECLARE @Failure_NotificationGroupID int 
			SELECT     @Failure_NotificationGroupID = Failure_NotificationGroupID
			FROM         QMS_CorrectiveActionEscalation
			WHERE     (CorrectiveActionEscalationID = @CorrectiveActionEscalationID)

			--if CA being set to Complete and a Failure_NotificationGroup has been set, reset NotificationGroup flag
			IF @CAStatusID = 4 --AND @Failure_NotificationGroupID IS NOT NULL --removed for TFS 7877 as this stops Completed alerts firing when no Failure_NotificationGroupID set
			BEGIN
				UPDATE	QMS_CorrectiveAction
				SET		NotificationSent = 0
				WHERE	(CorrectiveActionID = @CorrectiveActionID)
			END
			
				--if CA being set to Signed-off and AssignSupplierID has been set, reset NotificationGroup flag
			IF @CAStatusID IN (5, 6) AND @AssignSupplierID IS NOT NULL
			BEGIN
				UPDATE	QMS_CorrectiveAction
				SET		NotificationSent = 0
				WHERE	(CorrectiveActionID = @CorrectiveActionID)
			END
		END
	END
	SET NOCOUNT OFF

	--save data for CA to QMS_CorrectiveActionUser (should handle both new and updated CAs as @CorrectiveActionID is set either way)
    IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
        BEGIN --Add current user to current corrective action
            IF NOT EXISTS ( SELECT  CorrectiveActionID
                            FROM    dbo.QMS_CorrectiveActionUser
                            WHERE   UserID = @UserID
                                    AND CorrectiveActionID = @CorrectiveActionID )
                INSERT  INTO dbo.QMS_CorrectiveActionUser
                        ( CorrectiveActionID, UserID )
                VALUES  ( @CorrectiveActionID, -- CorrectiveActionID - int
                          @UserID -- UserID - int
                          )
        END
    ELSE
        EXEC SAVE_CorrectiveActionUser @CorrectiveActionID
	

	SELECT @CorrectiveActionID
END

GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SYNC_Import_AssessmentQuestionConstraint')
	EXEC ('CREATE PROCEDURE [dbo].[SYNC_Import_AssessmentQuestionConstraint] AS SELECT ''stub version, to be replaced''')
GO
ALTER PROCEDURE [dbo].[SYNC_Import_AssessmentQuestionConstraint] 
	@QuestionConstraintID int,
	@AssessmentQuestionID int,
	@Constraint_Text nvarchar(50),
	@Constraint_Text_LeftChars tinyint,
	@Constraint_Boolean bit,
	@Constraint_Date_Min datetime,
	@Constraint_Date_Max datetime,
	@Constraint_Date_TypeID int,
	@Constraint_Date_MinValue INT,
	@Constraint_Date_MaxValue INT,
	@Constraint_Numeric_Min float,
	@Constraint_Numeric_Max float,
	@Constraint_SourceID int,
	@QAFlagID int,
	@Score tinyint,
	@GuidanceID int,
	@ShortGuidance nvarchar(255),
	@CorrectiveActionTypeID int,
	@Constraint_Formula_Min nvarchar(1000) = NULL,
	@Constraint_Formula_Max nvarchar(1000) = NULL,
	@ShowHiddenSubQuestions BIT = NULL,
	@Sys_QuestionConstraintID int,
	@UserID int,
	@MachineName nvarchar(255),
	@RecordStatus tinyint,
	@Inserted datetime,
	@Updated datetime,
	@Deleted datetime,
	@GUID uniqueidentifier,
	@Created bigint,
	@RowVersion timestamp,
	@InsertCount int = 0 OUTPUT,
	@UpdateCount int = 0 OUTPUT,
	@ReceivedAnchor bigint = 0 OUTPUT
AS

IF @ShowHiddenSubQuestions IS NULL
	SET @ShowHiddenSubQuestions = 0

IF NOT EXISTS(SELECT * FROM QMS_AssessmentQuestionConstraint WHERE (QuestionConstraintID = @QuestionConstraintID))
BEGIN --Append record
	IF @RecordStatus < 201 AND EXISTS(SELECT AssessmentQuestionID FROM QMS_AssessmentQuestion WHERE (AssessmentQuestionID = @AssessmentQuestionID))
	BEGIN
		SET IDENTITY_INSERT QMS_AssessmentQuestionConstraint ON
		INSERT INTO QMS_AssessmentQuestionConstraint
                      (QuestionConstraintID, AssessmentQuestionID, Constraint_Text, Constraint_Text_LeftChars, Constraint_Boolean, Constraint_Date_Min, Constraint_Date_Max, 
                      Constraint_Date_TypeID, Constraint_Date_MinValue, Constraint_Date_MaxValue, Constraint_Numeric_Min, Constraint_Numeric_Max, Constraint_SourceID, QAFlagID, 
                      Score, GuidanceID, ShortGuidance, CorrectiveActionTypeID, Constraint_Formula_Min, Constraint_Formula_Max, Sys_QuestionConstraintID, UserID, MachineName, 
                      RecordStatus, Inserted, Updated, Deleted, GUID, ShowHiddenSubQuestions)
		VALUES     (@QuestionConstraintID,@AssessmentQuestionID,@Constraint_Text,@Constraint_Text_LeftChars,@Constraint_Boolean,@Constraint_Date_Min,@Constraint_Date_Max,@Constraint_Date_TypeID,@Constraint_Date_MinValue,@Constraint_Date_MaxValue,@Constraint_Numeric_Min,@Constraint_Numeric_Max,@Constraint_SourceID,@QAFlagID,@Score,@GuidanceID,@ShortGuidance,@CorrectiveActionTypeID,@Constraint_Formula_Min,@Constraint_Formula_Max,@Sys_QuestionConstraintID,@UserID,@MachineName,@RecordStatus,@Inserted,@Updated,@Deleted,@GUID, @ShowHiddenSubQuestions)
		SET @InsertCount = @@rowcount
		SET IDENTITY_INSERT QMS_AssessmentQuestionConstraint OFF
	END
END

ELSE
BEGIN --Update record
	UPDATE    QMS_AssessmentQuestionConstraint
	SET              Constraint_Text = @Constraint_Text, Constraint_Text_LeftChars = @Constraint_Text_LeftChars, Constraint_Boolean = @Constraint_Boolean, 
                      Constraint_Date_Min = @Constraint_Date_Min, Constraint_Date_Max = @Constraint_Date_Max, Constraint_Date_TypeID = @Constraint_Date_TypeID, 
                      Constraint_Date_MinValue = @Constraint_Date_MinValue, Constraint_Date_MaxValue = @Constraint_Date_MaxValue, 
                      Constraint_Numeric_Min = @Constraint_Numeric_Min, Constraint_Numeric_Max = @Constraint_Numeric_Max, Constraint_SourceID = @Constraint_SourceID, 
                      QAFlagID = @QAFlagID, Score = @Score, GuidanceID = @GuidanceID, ShortGuidance = @ShortGuidance, CorrectiveActionTypeID = @CorrectiveActionTypeID, 
                      Constraint_Formula_Min = @Constraint_Formula_Min, Constraint_Formula_Max = @Constraint_Formula_Max, 
                      Sys_QuestionConstraintID = @Sys_QuestionConstraintID, UserID = @UserID, MachineName = @MachineName, RecordStatus = @RecordStatus, 
                      Inserted = @Inserted, Updated = @Updated, Deleted = @Deleted, ShowHiddenSubQuestions = @ShowHiddenSubQuestions
	WHERE     (QuestionConstraintID = @QuestionConstraintID)
	SET @UpdateCount = @@rowcount
END

IF CAST(@RowVersion AS bigint) > @Created
	SET @ReceivedAnchor = CAST(@RowVersion AS bigint)
ELSE
	SET @ReceivedAnchor = @Created
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_CorrectiveActionUser')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_CorrectiveActionUser] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Jon Edwards
-- Create date: 06/01/2015
-- Description:	Saves CA user data to QMS_CorrectiveActionUser
-- Version:		3
-- =============================================
-- JE TFS 13-12635: created
-- JE TFS 13-13404: amended INSERT for unassigned users into QMS_CorrectiveActionUser
-- JT TFS 13-14013: Simplified code, and remove corrective actions that you no longer have permission to see.
-- =============================================
ALTER PROCEDURE SAVE_CorrectiveActionUser
	@CorrectiveActionID int
AS
BEGIN
	SET NOCOUNT ON

	IF nullif(object_id('tempdb..#tmpCorrectiveActionUser'), 0) IS NOT NULL DROP TABLE #tmpCorrectiveActionUser
	CREATE TABLE #tmpCorrectiveActionUser (
		UserID int,
		Assigned bit
	)

	-- We want #tmpCorrectiveActionUser to contain exactly what QMS_CorrectiveAction should eventually contain.
	INSERT INTO #tmpCorrectiveActionUser(UserID, Assigned)
	SELECT UserID, 1
	FROM dbo.FNC_GENERATE_CorrectiveAction_AssignUsers()
	WHERE CorrectiveActionID = @CorrectiveActionID 

	INSERT INTO #tmpCorrectiveActionUser (UserID, Assigned)
	SELECT UserID, 0
	FROM dbo.FNC_GENERATE_CorrectiveAction_UserList(@CorrectiveActionID) as userlist
	WHERE UserID NOT IN (SELECT UserID FROM #tmpCorrectiveActionUser)
	
	-- UPDATE EXISTING
	UPDATE QMS_CorrectiveActionUser
	SET Assigned = CAUser.Assigned,
		Updated = GETUTCDATE()
	FROM QMS_CorrectiveActionUser
	INNER JOIN #tmpCorrectiveActionUser as CAUser ON CAUser.UserID = QMS_CorrectiveActionUser.UserID
													AND QMS_CorrectiveActionUser.CorrectiveActionID = @CorrectiveActionID 
	WHERE QMS_CorrectiveActionUser.Assigned <> CAUser.Assigned


	-- INSERT MISSING
	INSERT QMS_CorrectiveActionUser (CorrectiveActionID, UserID, Assigned)
	SELECT @CorrectiveActionID, CAUser.UserID, CAUser.Assigned 
	FROM #tmpCorrectiveActionUser AS CAUser
	WHERE NOT EXISTS (	SELECT 1 FROM QMS_CorrectiveActionUser
						WHERE CorrectiveActionID = @CorrectiveActionID 
						AND UserID = CAUser.UserID)


	-- DELETE REMOVED ONES
	DELETE QMS_CorrectiveActionUser
	WHERE CorrectiveActionID = @CorrectiveActionID
	AND UserID NOT IN (SELECT UserID FROM #tmpCorrectiveActionUser)
	
	DROP TABLE #tmpCorrectiveActionUser

	SET NOCOUNT OFF
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Response_PreviousList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Response_PreviousList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 28/07/2008
-- Description:	Gets list of previous responses
-- Version:		22
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- Jonathan Toolan: 26-Mar-2012 Location Added.
-- Jonathan Toolan: TFS 3211 Multiple AssessmentTypeID_LIST
-- JE TFS 6272 - added Child Check/Top Level Check functionality
-- JT TFS 7346 - Added AssessmentID filter parameter.
-- JT TFS 7989 - Added ExportSetupTable to this SP as an example, so that custom SPs can copy and modify more easily.
-- JT TFS 8130 - ShowOutstanding to switch between "outstanding" mode and "completed" mode.
-- JT TFS 8578 - Added generic filtering if applied.
-- JE TFS 13-466: added provision for AQC Multi assessment format
-- JT TFS 13-612: Added DisplayMode parameter to show multi-check children (default) or parents.
-- JE TFS 13-1621: amended setting of @multiCheckDisplay so parent checks only are shown when isViewQCChecksScreen
-- JE TFS 13-2537: amended so that 3:isQCActionsScreen shows Parent checks only
-- JF TFS 2880 - Ensure that Child checks are available in Office Web - View QC Checks. 
				--Basically if the Multicheck contains a Summary column, then I would expect the Parent check to be referenced/displayed in ALL views
				--If no Summary Column (majority of multicheck customers) - then the Child check should be referenced/displayed in ALL views
-- TFS 2955		Navigate to Tasks > QC Actions for an assessment type of Active Quality Chain,Expected outcome: I am expecting to see the Parent checks here, Actual outcome: I am seeing the Child Checks
-- JE TFS 13-9699: only show non-Approved child checks for AQC assessment type
-- JE TFS 13-11892: output column indicating what sort of multi check response  is (i.e. parent or child) for AQC
-- JE TFS 13-12540: AQC Multi Check shows parent checks only for Previous tab
-- MP TFS 13-12735: Added logic to check for AQC approval config setting
-- jf 13042 - AQC may not be multicheck so allow for it
-- PH TFS 13-15140:  Added 'IsWebChecks' column to determine whether the Response is authorised to be edited using WebChecks front end (permission derived from APP_Menu with ParentID = 233)
-- jf tfs 16880 -- removed inner joins to FNC_User_ResponseList and used TEMP_selected instead - UBE Check view performance for user with 2000+ stock code permissions
				-- took resource string calcs out of nexted select and put in main output query (QAFlagID, ResourceName,QAFlag)
-- =============================================
ALTER PROCEDURE QRY_Response_PreviousList
    @AssessmentTypeID INT = NULL ,
    @AssessmentTypeID_LIST NVARCHAR(255) = NULL ,
    @DateFrom DATETIME ,
    @DateTo DATETIME ,
    @UserID INT ,
    @AssessmentID INT = NULL ,
    @ShowOutstanding BIT = 0 , -- 0: show completed checks only, 1: show outstanding checks only, NULL: show all checks regardless of ResponseStatus.
    @IsFilterApplied BIT = 0 ,
    @DisplayMode INT = NULL	
		-- 0 or NULL: not specified.
		-- 1:isPreviousChecksScreen
		-- 2:isViewQCChecksScreen
		-- 3:isQCActionsScreen
		-- 4:isQCCheckReport
AS 
    BEGIN
        SET NOCOUNT ON
--exec QRY_Response_PreviousList @AssessmentTypeID=13,@AssessmentTypeID_LIST=default,@DateFrom='2015-02-19 00:00:00',@DateTo='2015-03-19 23:59:59',@UserID=100,@AssessmentID=default,@ShowOutstanding=NULL,@IsFilterApplied=0,@DisplayMode=2    
-- If (isPreviousChecksScreen), then show parent checks (1), otherwise show children (2).
-- @multiCheckDisplay should match return value from dbo.FNC_Response_IsMultiCheck.

--FNC_Response_IsMultiCheck results
--0 -- Isn't multi-check at all.
--1 -- Is a parent multi-check
--2 -- Is a child multi-check
																				
        

--get Assessment Format
        DECLARE @AssessmentFormat INT
        SELECT  @AssessmentFormat = AssessmentFormat
        FROM    APP_AssessmentType
        WHERE   AssessmentTypeID = @AssessmentTypeID


		DECLARE @multiCheckDisplay INT;
        SELECT  @multiCheckDisplay = CASE WHEN ISNULL(@DisplayMode, 0) IN ( 1 ) 
										THEN  1    --show parent checks only for previouschecks screen
										WHEN @AssessmentFormat = 6 THEN 1
										ELSE 2
									END
									
		DECLARE @AQCApproval bit
		SELECT @AQCApproval = ISNULL(dbo.FNC_GetOptionValue(120),0)

        IF ( @AssessmentTypeID_LIST IS NULL ) 
            SET @AssessmentTypeID_LIST = CONVERT(NVARCHAR, @AssessmentTypeID)

--Clear temp table
        DELETE  FROM TEMP_Selected
        WHERE   ( UserID = @UserID )

--Fill temp table
        IF @IsFilterApplied = 0 
            BEGIN
                INSERT INTO TEMP_Selected
                      (SelectedID, UserID)
				SELECT     QMS_Response.ResponseID, @UserID AS UserID
				FROM         QMS_Response INNER JOIN
                      dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
                      QMS_Response.ResponseID = User_ResponseList.ResponseID INNER JOIN
                      dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',') AS SplitString ON QMS_Response.AssessmentTypeID = SplitString.Token

            END
        ELSE 
            BEGIN
                INSERT INTO TEMP_Selected
                      (SelectedID, UserID)
				SELECT     ResponseID, @UserID AS UserID
				FROM         TEMP_ResponseFilter
				WHERE     (UserID = @UserID)

            END

        SET NOCOUNT OFF

        IF @AssessmentTypeID_LIST = '2' --Arrival check
            BEGIN
                SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, QMS_Assessment.AssessmentName, ResponseHeader.Supplier, ResponseHeader.StockCode, 
                      ResponseHeader.Location, QMS_Response.StartDate, QMS_Response.EndDate, QMS_Response.Actual_Score, QMS_Response.Actual_QAFlagID, 
					  --QAFlag.QAFlag, 
					  ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) as QAFlag,
                      QMS_Response.Override_QAFlagID, 
					  --QAFlag_Override.QAFlag AS Override_QAFlag, 
					  ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) as Override_QAFlag,
					  QMS_Response.Offline_ResponseID, 
                      CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, QMS_User.FullName
				FROM         QMS_Response INNER JOIN
                      TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
                      --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
                      QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
                      QMS_User ON QMS_Response.UserID = QMS_User.UserID --INNER JOIN
                      --dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
                      --QMS_Response.ResponseID = User_ResponseList.ResponseID 
					  LEFT OUTER JOIN
                      dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
                          (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
                            FROM          QMS_Attachment
                            WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
                            GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ResourceName,QAFlag
                            FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ResourceName,QAFlag
                            FROM          APP_QAFlag AS APP_QAFlag_1) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
				WHERE     
				
					(QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND 
					(@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL) AND 
					(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND 
					(@ShowOutstanding IS NULL) 
					OR                      
					(QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND 
					(@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL) AND 
					(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND 
					(@ShowOutstanding = 1) AND 
                    (QMS_Response.ResponseStatus = 0) 
                    OR
                    (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND 
                    (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL) AND 
                    (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND 
                    (@ShowOutstanding = 0) AND 
                    (QMS_Response.ResponseStatus <> 0)
                    
				ORDER BY QMS_Response.EndDate DESC
            END
        ELSE 
            IF @AssessmentTypeID_LIST IN ( '3', '4', '5' ) --Online/Label/Dispatch
                BEGIN
                    SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, ResponseHeader.Customer, ResponseHeader.StockCode, ResponseHeader.Location, 
                      ResponseHeader.DepotDate AS [Depot date], QMS_Assessment.AssessmentName, QMS_Response.StartDate, QMS_Response.EndDate, 
                      QMS_Response.Actual_QAFlagID, 
					  --QAFlag.QAFlag, 
					  ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) as QAFlag,
					  QMS_Response.Override_QAFlagID, 
					  --QAFlag_Override.QAFlag AS Override_QAFlag, 
					  ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) as Override_QAFlag,
                      QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
                      QMS_User.FullName
					FROM         QMS_Response INNER JOIN
                      TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
                      --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
                      QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
                      QMS_User ON QMS_Response.UserID = QMS_User.UserID --INNER JOIN
                      --dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
                      --QMS_Response.ResponseID = User_ResponseList.ResponseID 
					  LEFT OUTER JOIN
                      dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID 
					  LEFT OUTER JOIN
                          (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
                            FROM          QMS_Attachment
                            WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
                            GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ResourceName,QAFlag
                            FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ResourceName,QAFlag
                            FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
					WHERE     (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
                      @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding IS NULL) OR
                      (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
                      @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 1) AND 
                      (QMS_Response.ResponseStatus = 0) OR
                      (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
                      @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 0) AND 
                      (QMS_Response.ResponseStatus <> 0)
					ORDER BY QMS_Response.EndDate DESC
                END
            ELSE 
                IF @AssessmentTypeID_LIST = '6' --Shelf life
                    BEGIN
                        SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, ResponseHeader.Customer, ResponseHeader.StockCode, ResponseHeader.Location, 
							  ResponseHeader.AssessmentSubType AS [Test type], ResponseHeader.DepotDate AS [Depot date], QMS_Assessment.AssessmentName, QMS_Response.StartDate, 
							  QMS_Response.EndDate, QMS_Response.Actual_QAFlagID, 
							  --QAFlag.QAFlag, 
							  ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) as QAFlag,
							  QMS_Response.Override_QAFlagID, 
							  --QAFlag_Override.QAFlag AS Override_QAFlag, 
							  ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) as Override_QAFlag,
							  QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
							  QMS_User.FullName
						FROM         QMS_Response INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
							  --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID --INNER JOIN
							  --dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  --QMS_Response.ResponseID = User_ResponseList.ResponseID 
							  LEFT OUTER JOIN
							  dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
								  (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
									FROM          QMS_Attachment
									WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
									GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ResourceName,QAFlag
									FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ResourceName,QAFlag
									FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
						WHERE     (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding IS NULL) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 1) AND 
							  (QMS_Response.ResponseStatus = 0) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 0) AND 
							  (QMS_Response.ResponseStatus <> 0)
						ORDER BY QMS_Response.EndDate DESC
                    END
            ELSE 
                IF @AssessmentTypeID_LIST = '16' --Active Quality Chain
                    BEGIN
                        SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, ResponseHeader.Customer, ResponseHeader.StockCode, ResponseHeader.Location, 
							  ResponseHeader.AssessmentSubType AS [Test type], ResponseHeader.DepotDate AS [Depot date], QMS_Assessment.AssessmentName, QMS_Response.StartDate, 
							  QMS_Response.EndDate, QMS_Response.Actual_QAFlagID, 
							  --QAFlag.QAFlag, 
							  ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) as QAFlag,
							  QMS_Response.Override_QAFlagID, 
							  --QAFlag_Override.QAFlag AS Override_QAFlag, 
							  ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) as Override_QAFlag,
							  QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
							  QMS_User.FullName, CAST(CASE WHEN @AQCApproval = 1 THEN ISNULL(dbo.FNC_Response_IsApproved(QMS_Response.ResponseID), 0) ELSE 1 END AS BIT) AS IsApproved,
							  ISNULL(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID), 0) AS MultiCheckLevel
						FROM         QMS_Response INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID --INNER JOIN
							  --dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  --QMS_Response.ResponseID = User_ResponseList.ResponseID 
							  LEFT OUTER JOIN
							  dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
								  (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
									FROM          QMS_Attachment
									WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
									GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ResourceName,QAFlag
									FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
								  (SELECT    QAFlagID, ResourceName,QAFlag
									FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
						WHERE (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST))
								AND ((@multiCheckDisplay = 1 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1) OR (@multiCheckDisplay = 2 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)) OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0)
								AND (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL)
								AND (@ShowOutstanding IS NULL) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST))
								AND ((@multiCheckDisplay = 1 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1) OR (@multiCheckDisplay = 2 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)) OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0)
								AND (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL)
								AND (@ShowOutstanding = 1)
								AND (QMS_Response.ResponseStatus = 0) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST))
								AND ((@multiCheckDisplay = 1 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1) OR (@multiCheckDisplay = 2 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)) OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0)
								AND (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL)
								AND (@ShowOutstanding = 0)
								AND (QMS_Response.ResponseStatus <> 0)
						ORDER BY QMS_Response.EndDate DESC
                    END
                ELSE 
                    BEGIN
                        SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, QMS_Assessment.AssessmentName, ResponseHeader.Location, QMS_Response.StartDate, 
							  QMS_Response.EndDate, QMS_Response.Actual_Score, QMS_Response.Actual_QAFlagID, 
							  
							  --QAFlag.QAFlag as QAFlag,
							  ISNULL(dbo.FNC_ResourceString_User(QAFlag.ResourceName, @UserID), QAFlag.QAFlag) as QAFlag, 							  
							  QMS_Response.Override_QAFlagID, 
							  
							  --QAFlag_Override.QAFlag AS Override_QAFlag, 
							  ISNULL(dbo.FNC_ResourceString_User(QAFlag_Override.ResourceName, @UserID), QAFlag_Override.QAFlag) as Override_QAFlag, 

							  
							  QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) 
							  = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, QMS_User.FullName
						FROM         QMS_Response INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
							  --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID --INNER JOIN
							  --dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  --QMS_Response.ResponseID = User_ResponseList.ResponseID 
							  LEFT OUTER JOIN
								  (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
									FROM          QMS_Attachment
									WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
									GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
								  (SELECT     
								  QAFlagID, 
								  --ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
								  ResourceName,
								  QAFlag
									FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
								  (SELECT     
								  QAFlagID, 
								  --ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
								  QAFlag,ResourceName

									FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID LEFT OUTER JOIN
							  dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID
						WHERE     (QMS_Response.AssessmentTypeID IN
								  (SELECT     CONVERT(INT, Token) AS Expr1
									FROM          dbo.FNC_SplitString(@AssessmentTypeID_LIST, ','))) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding IS NULL) OR
							  (QMS_Response.AssessmentTypeID IN
								  (SELECT     CONVERT(INT, Token) AS Expr1
									FROM          dbo.FNC_SplitString(@AssessmentTypeID_LIST, ','))) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 1) AND 
							  (QMS_Response.ResponseStatus = 0) OR
							  (QMS_Response.AssessmentTypeID IN
								  (SELECT     CONVERT(INT, Token) AS Expr1
									FROM          dbo.FNC_SplitString(@AssessmentTypeID_LIST, ','))) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 0) AND 
							  (QMS_Response.ResponseStatus <> 0)
						ORDER BY QMS_Response.EndDate DESC
                    END

--get list of columns that need to be hidden in Excel export
--NOTE:		tableindex is important, and zero based. This export table is table 1 because it's the second table returned in the dataset
--		FixedColumns: These are returned in the main body. These should be fixed. Leave this as NULL to use default ones specified by front end.
--		StaticColumns: These are displayed underneath the compact text in the grid, and do not move, but are specific for each customer.
--		CyclicColumns: If this contains the word "REMAINING_COLUMNS", then it will also include any columns not specified in Fixed or Static.
-- NOTE:	leabing static and cyclic columns NULL will use first two columns not in the default fixed columns as "static", and other columns not in either fixed or static as "cyclic"
        SELECT  1 AS TableIndex ,
                CONVERT(BIT, 1) AS ExportSetupTable ,
                'Portrait/Landscape' AS Orientation ,
                'n/a' AS SheetName ,
                'n/a' AS HiddenColumns ,
                'n/a' AS FixedColumns ,
                'n/a' AS StaticColumns ,
                'n/a' AS CyclicColumns
        UNION
        SELECT  0 ,
                0 ,
                'Portrait' ,
                'View Checks' ,
                NULL ,
                NULL ,
                'Location,FullName' ,
                'REMAINING_COLUMNS'
        ORDER BY TableIndex ASC

    END
GO



--PRINT 'Updating sync procedures...'

--PRINT 'Data migration tasks...'

PRINT 'Database updated successfully!'
GO
