-- =============================================
-- Author:		Jon Edwards
-- Create date: 07/01/2015
-- Version:		0.1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 56
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO

--PRINT 'Inserting tables...'

--PRINT 'Updating tables...'
IF NOT EXISTS(SELECT * FROM sys.columns where name = 'IsRequiredForSignOff' and object_id in( SELECT object_id FROM SYS.objects where name = 'APP_UserLevel' ))
alter table APP_UserLevel add
  IsRequiredForSignOff bit not null constraint DF_APP_UserLevel_IsRequiredForSignOff default ((0));
GO


IF NOT EXISTS(SELECT * from sys.Columns where is_nullable = 1 and name  = 'SupplierID' and object_id in (SELECT object_id from sys.objects where name  = 'QMS_SupplierResponse' and type = 'u'))
ALTER table dbo.QMS_SupplierResponse alter column
  SupplierID int
GO


--PRINT 'Creating indexes...'

--PRINT 'Updating Views...'

--PRINT 'Dropping Views...'

PRINT 'Updating User Defined Functions...'
GO

/****** Object:  UserDefinedFunction [dbo].[FNC_Assessment_CustomerID]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_Assessment_CustomerID]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
DROP FUNCTION [dbo].[FNC_Assessment_CustomerID]
GO
/****** Object:  UserDefinedFunction [dbo].[FNC_User_ResponseList]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_User_ResponseList]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
DROP FUNCTION [dbo].[FNC_User_ResponseList]
GO
/****** Object:  UserDefinedFunction [dbo].[FNC_ResponseQuestion_Selected]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_ResponseQuestion_Selected]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
DROP FUNCTION [dbo].[FNC_ResponseQuestion_Selected]
GO
/****** Object:  UserDefinedFunction [dbo].[FNC_GENERATE_CorrectiveAction_UserList]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_GENERATE_CorrectiveAction_UserList]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
DROP FUNCTION [dbo].[FNC_GENERATE_CorrectiveAction_UserList]
GO
/****** Object:  UserDefinedFunction [dbo].[FNC_Assessment_IsSignedOffByUserLevel]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_Assessment_IsSignedOffByUserLevel]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
DROP FUNCTION [dbo].[FNC_Assessment_IsSignedOffByUserLevel]
GO

/****** Object:  UserDefinedFunction [dbo].[FNC_Assessment_IsSignedOffByUserLevel]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_Assessment_IsSignedOffByUserLevel]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
BEGIN
execute dbo.sp_executesql @statement = N'-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 12/01/2015
-- Description:	Has the assessment been signed off by that UserLevel since the last rejection (or ever if no rejections)?
-- Version:		1
-- =============================================
-- JT TFS 13-11815: Created
-- =============================================
CREATE FUNCTION [dbo].[FNC_Assessment_IsSignedOffByUserLevel](@AssessmentID int, @UserLevelID int)
RETURNS bit
AS
BEGIN
	DECLARE @tmpRecordLock TABLE (
		AssessmentID int,
		UserLevel int,
		LockStatus int,
		LockStart datetime
	)

	INSERT @tmpRecordLock (AssessmentID, UserLevel, LockStatus, LockStart)
	SELECT assessment.AssessmentID, dbo.FNC_GetUserLevel(lock.UserID) UserLevel, LockStatus, LockStart 
	FROM QMS_RecordLock lock
	INNER JOIN QMS_Assessment assessment ON assessment.GUID = lock.RowGUID
	WHERE lock.ObjectName = ''QMS_Assessment''
	AND assessment.AssessmentID = @AssessmentID
	AND LockStatus IN (111, 120, 121)

	-- Need to consider what happens if it''s rejected. It''ll need a new sign-off from everyone.
	IF EXISTS (SELECT * FROM @tmpRecordLock WHERE LockStatus = 111)
		DELETE @tmpRecordLock
		WHERE LockStart <= (SELECT max(LockStart) FROM @tmpRecordLock WHERE LockStatus = 111)

	RETURN CASE WHEN EXISTS (SELECT * FROM @tmpRecordLock WHERE UserLevel = @UserLevelID) THEN 1 ELSE 0 END
END
' 
END

GO
/****** Object:  UserDefinedFunction [dbo].[FNC_GENERATE_CorrectiveAction_UserList]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_GENERATE_CorrectiveAction_UserList]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
BEGIN
execute dbo.sp_executesql @statement = N'-- =============================================
-- Author:		Jon Edwards
-- Create date: 06/01/2015
-- Description:	Returns list of users with visibility of a Corrective Action
-- Version: 2
-- =============================================
-- JE TFS 13-9973: created
-- JE TFS 13-14013: corrected returning of users with no Location permissions
-- =============================================
CREATE FUNCTION [dbo].[FNC_GENERATE_CorrectiveAction_UserList]
(
	@CorrectiveActionID int
)
RETURNS
@Users TABLE
(
	UserID int
)
AS
BEGIN

	--get CA''s LocationID
	DECLARE @LocationID int
	SELECT @LocationID = LocationID FROM QMS_CorrectiveAction WHERE CorrectiveActionID = @CorrectiveActionID

	--get CA''s SupplierID
	DECLARE @SupplierID int
	SELECT @SupplierID = AssignSupplierID FROM QMS_CorrectiveAction WHERE CorrectiveActionID = @CorrectiveActionID

	--get StockCodeID for CA linked at check level
	DECLARE @StockCodeID int
	SELECT	@StockCodeID = QMS_ResponseHeader.Header_SourceID
	FROM	QMS_ResponseHeader
			INNER JOIN QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			INNER JOIN QMS_CorrectiveAction ON QMS_ResponseHeader.ResponseID = QMS_CorrectiveAction.SourceObjectID
	WHERE	(QMS_CorrectiveAction.CorrectiveActionID = @CorrectiveActionID)
			AND (QMS_CorrectiveAction.CASourceID = 1)
			AND (QMS_AssessmentHeader.DataTypeID = 9)
			AND (QMS_AssessmentHeader.DataLookupID = 2)

	--get StockCodeID for CA linked at question level if CA not linked at check level
	IF @StockCodeID IS NULL
		SELECT	@StockCodeID = QMS_ResponseHeader.Header_SourceID
		FROM	QMS_ResponseQuestion
				INNER JOIN QMS_ResponseHeader ON QMS_ResponseQuestion.ResponseID = QMS_ResponseHeader.ResponseID
				INNER JOIN QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID		
				INNER JOIN QMS_CorrectiveAction ON QMS_ResponseQuestion.ResponseQuestionID = QMS_CorrectiveAction.SourceObjectID
		WHERE	(QMS_CorrectiveAction.CorrectiveActionID = @CorrectiveActionID)
				AND (QMS_CorrectiveAction.CASourceID = 2)
				AND (QMS_AssessmentHeader.DataTypeID = 9)
				AND (QMS_AssessmentHeader.DataLookupID = 2)

	--get BusinessGroup and StockCodeSubGroup for CA''s StockCodeID (if CA has one)
	IF @StockCodeID IS NOT NULL
	BEGIN
		DECLARE @BusinessGroup AS nvarchar(50)
		SELECT @BusinessGroup = BusinessGroup FROM PM_VW_StockCode WHERE StockCodeID = @StockCodeID

		DECLARE @StockCodeSubGroup AS nvarchar(50)
		SELECT @StockCodeSubGroup = StockCodeSubGroup FROM PM_VW_StockCode WHERE StockCodeID = @StockCodeID
	END

	--get users with permissions to the LocationID
	INSERT INTO @Users
	SELECT	QMS_User.UserID
	FROM	QMS_User
			INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	(PermissionType = 103) AND (PermissionAllow = 1) AND (PermissionTitle = @LocationID)

	--get users with permissions to the SupplierID
	INSERT INTO @Users
	SELECT	QMS_User.UserID
	FROM	QMS_User
			INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	(PermissionType = 101) AND (PermissionAllow = 1) AND (PermissionTitle = @SupplierID)
			AND NOT EXISTS (SELECT UserID FROM @Users WHERE UserID = QMS_User.UserID)

	--get users with permissions to the StockCodeID (need to check three levels of hierarchy: Business Group, Sub Group and Stock Code)
	INSERT INTO @Users
	SELECT	QMS_User.UserID
	FROM	QMS_User
			INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	((PermissionType = 106) AND (PermissionAllow = 1) AND (PermissionTitle = @BusinessGroup))
			AND NOT EXISTS (SELECT UserID FROM @Users WHERE UserID = QMS_User.UserID)

	INSERT INTO @Users
	SELECT	QMS_User.UserID
	FROM	QMS_User
			INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	((PermissionType = 105) AND (PermissionAllow = 1) AND (PermissionTitle = @StockCodeSubGroup))
			AND NOT EXISTS (SELECT UserID FROM @Users WHERE UserID = QMS_User.UserID)

	INSERT INTO @Users
	SELECT	QMS_User.UserID
	FROM	QMS_User
			INNER JOIN QMS_UserPermissions ON QMS_User.UserID = QMS_UserPermissions.UserID
	WHERE	((PermissionType = 104) AND (PermissionAllow = 1) AND (PermissionTitle = @StockCodeID))
			AND NOT EXISTS (SELECT UserID FROM @Users WHERE UserID = QMS_User.UserID)

	--table of all users with Location permissions
	DECLARE @LocationPermissions TABLE (UserID int, LocationID int)
	INSERT INTO @LocationPermissions
	SELECT DISTINCT	UserID, PermissionTitle
	FROM	QMS_UserPermissions
	WHERE	(PermissionType = 103) AND (PermissionAllow = 1)

	--get admin users or users with no specific Location permissions as they still have visibility of it
	INSERT INTO @Users
	SELECT DISTINCT	QMS_User.UserID
	FROM	QMS_User
			LEFT OUTER JOIN @LocationPermissions AS LocationPermissions ON QMS_User.UserID = LocationPermissions.UserID
	WHERE	(QMS_User.UserLevel = 1) --admin can see all CAs
			OR
			((QMS_User.UserLevel <> 1) AND (LocationPermissions.UserID IS NULL)
				AND (@StockCodeID IS NULL) AND (@BusinessGroup IS NULL) AND (@StockCodeSubGroup IS NULL)) --show CAs to non-admin users if they have NO location permissions and the CA has NO StockCodeID assigned to it
			OR
			((QMS_User.UserLevel <> 1) AND (@LocationID IS NULL)) --show CAs to non-admin users if they have Location permissons but the CA has NO LocationID
			AND NOT EXISTS (SELECT UserID FROM @Users WHERE UserID = QMS_User.UserID)

	RETURN

END
' 
END

GO
/****** Object:  UserDefinedFunction [dbo].[FNC_ResponseQuestion_Selected]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_ResponseQuestion_Selected]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
BEGIN
execute dbo.sp_executesql @statement = N'-- =============================================
-- Author:		Markp
-- Create date: 13/05/2009
-- Description:	Gets response questions in to flat table form
-- Version: 7
-- =============================================
-- TFS 8116: Converted from cursor, and return "Issue1 (comment), Issue2, Issue3 (comment)" instead of "Issue1 (comment), Issue3 (comment)" for comments.
-- TFS 8724 - added @ShowQuestionLevelOnReports filtering
-- tfs 11643 - Include question responses in the QC Summary Report comments column
-- =============================================
create FUNCTION [dbo].[FNC_ResponseQuestion_Selected] 
(
	@UserID int,
	@QAFlagID int=0
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	GreenResponses nvarchar(1000),
	GreenComments nvarchar(1000),
	AmberResponses nvarchar(1000),
	AmberComments nvarchar(1000),
	RedResponses nvarchar(1000),
	RedComments nvarchar(1000),
	GreenLevel nvarchar(255),
	AmberLevel nvarchar(255),
	RedLevel nvarchar(255)
)
AS



BEGIN
	DECLARE @ShowQuestionLevelOnReports BIT
	SET @ShowQuestionLevelOnReports = CAST(ISNULL(dbo.FNC_GetOptionValue(109),0) AS bit)


	DECLARE @MaxResponseLen INT
	SET @MaxResponseLen=995 --needs to be slightly less than 1000 to allow for additional characters
	
		DECLARE @ResponseMatches TABLE (
		ResponseID int,
		QAFlagID int,
		Question nvarchar(255),
		Comments nvarchar(255),
		Response_Text nvarchar(255)
	)
	INSERT @ResponseMatches (ResponseID, QAFlagID, Question, Comments, QMS_ResponseQuestion.Response_Text)
	SELECT QMS_ResponseQuestion.ResponseID, QMS_ResponseQuestion.Recommended_QAFlagID, QMS_QuestionPool.Question, QMS_ResponseQuestion.QuestionComments, QMS_ResponseQuestion.Response_Text
	FROM         QMS_QuestionPool 
	INNER JOIN	QMS_AssessmentQuestion ON QMS_QuestionPool.QuestionID = QMS_AssessmentQuestion.QuestionID 
	INNER JOIN	QMS_ResponseQuestion ON QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID 
	INNER JOIN	TEMP_Selected ON QMS_ResponseQuestion.ResponseID = TEMP_Selected.SelectedID AND	TEMP_Selected.UserID = @UserID
	WHERE     (QMS_ResponseQuestion.RecordStatus < 201) 
	AND		(QMS_AssessmentQuestion.RecordStatus < 201) 
	AND		(QMS_ResponseQuestion.Recommended_QAFlagID IS NOT NULL) 
	AND (	
			(@QAFlagID > 0 AND QMS_ResponseQuestion.Recommended_QAFlagID = @QAFlagID) 
		OR
			(@QAFlagID = 0) 
		OR	
			(@QAFlagID = -2 AND QMS_ResponseQuestion.Recommended_QAFlagID IN (2, 3))
	)
	ORDER BY QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.QuestionNo
	
		DECLARE @ResponseLevels TABLE (
		ResponseID int,
		QuestionLevel nvarchar(50),
		QAFlagID int
	)
	
	IF @ShowQuestionLevelOnReports = 1
	
		INSERT @ResponseLevels (ResponseID, QuestionLevel, QAFlagID)
		SELECT     ResponseSummary.ResponseID, APP_QuestionLevel.QuestionLevel, QMS_AssessmentConstraint.QAFlagID
		FROM       APP_AssessmentConstraintType INNER JOIN
						  APP_QuestionLevel ON APP_AssessmentConstraintType.QuestionLevelID = APP_QuestionLevel.QuestionLevelID INNER JOIN
						  QMS_AssessmentConstraint ON 
						  APP_AssessmentConstraintType.AssessmentConstraintTypeID = QMS_AssessmentConstraint.AssessmentConstraintTypeID INNER JOIN
							  (SELECT     QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.AssessmentID, QMS_AssessmentQuestion.QuestionLevelID, 
													   SUM(QMS_ResponseQuestion.Score) AS LevelScore
								FROM          QMS_ResponseQuestion INNER JOIN
													   QMS_Response ON QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID INNER JOIN
													   QMS_AssessmentQuestion ON QMS_ResponseQuestion.AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID INNER JOIN
													   TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
								WHERE      (QMS_ResponseQuestion.RecordStatus < 201) AND (TEMP_Selected.UserID = @UserID) AND (@QAFlagID > 0) AND 
													   (QMS_Response.Actual_QAFlagID = @QAFlagID) OR
													   (QMS_ResponseQuestion.RecordStatus < 201) AND (TEMP_Selected.UserID = @UserID) AND (@QAFlagID = 0) OR
													   (QMS_ResponseQuestion.RecordStatus < 201) AND (TEMP_Selected.UserID = @UserID) AND (@QAFlagID = - 2) AND 
													   (QMS_Response.Actual_QAFlagID = 3 OR
													   QMS_Response.Actual_QAFlagID = 2)
								GROUP BY QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.AssessmentID, QMS_AssessmentQuestion.QuestionLevelID
								HAVING      (SUM(QMS_ResponseQuestion.Score) IS NOT NULL) OR
													   (SUM(QMS_ResponseQuestion.Score) IS NOT NULL) OR
													   (SUM(QMS_ResponseQuestion.Score) IS NOT NULL)) AS ResponseSummary ON 
						  QMS_AssessmentConstraint.AssessmentID = ResponseSummary.AssessmentID AND 
						  APP_AssessmentConstraintType.QuestionLevelID = ResponseSummary.QuestionLevelID
		WHERE     (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NOT NULL) AND 
						  (QMS_AssessmentConstraint.Constraint_Score_Max IS NOT NULL) AND (ResponseSummary.LevelScore >= QMS_AssessmentConstraint.Constraint_Score_Min) AND 
						  (ResponseSummary.LevelScore <= QMS_AssessmentConstraint.Constraint_Score_Max) OR
						  (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NULL) AND 
						  (QMS_AssessmentConstraint.Constraint_Score_Max IS NOT NULL) AND (ResponseSummary.LevelScore <= QMS_AssessmentConstraint.Constraint_Score_Max) OR
						  (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NOT NULL) AND 
						  (QMS_AssessmentConstraint.Constraint_Score_Max IS NULL) AND (ResponseSummary.LevelScore >= QMS_AssessmentConstraint.Constraint_Score_Min)
		ORDER BY ResponseSummary.ResponseID, APP_AssessmentConstraintType.QuestionLevelID
	
	
	INSERT INTO @Response (ResponseID, GreenResponses, GreenComments, AmberResponses, AmberComments, RedResponses, RedComments, GreenLevel, AmberLevel, RedLevel) 
	SELECT	ResponseID, 
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + Question  +  CASE WHEN Response_Text IS NULL THEN '''' ELSE '' ('' + Response_Text + '') '' END
						FROM @ResponseMatches WHERE QAFlagID = 1 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 1000)) AS GreenResponses,
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + Question + CASE WHEN Comments IS NULL THEN '''' ELSE '' ('' + Comments + '')'' END 
						FROM @ResponseMatches WHERE QAFlagID = 1 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 1000)) AS GreenComments,	
						
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + Question +  CASE WHEN Response_Text IS NULL THEN '''' ELSE '' ('' + Response_Text + '') '' END
						FROM @ResponseMatches WHERE QAFlagID = 2 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 1000)) AS AmberResponses,
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + Question  + CASE WHEN Comments IS NULL THEN '''' ELSE '' ('' + Comments + '')'' END 
						FROM @ResponseMatches WHERE QAFlagID = 2 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 1000)) AS AmberComments,
						
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + Question +  CASE WHEN Response_Text IS NULL THEN '''' ELSE '' ('' + Response_Text + '') '' END
						FROM @ResponseMatches WHERE QAFlagID = 3 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 1000)) AS RedResponses,
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + Question  + CASE WHEN Comments IS NULL THEN '''' ELSE '' ('' + Comments + '')'' END 
						FROM @ResponseMatches WHERE QAFlagID = 3 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 1000)) AS RedComments,
						
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + QuestionLevel
						FROM @ResponseLevels WHERE QAFlagID = 1 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 255)) AS GreenLevel,
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + QuestionLevel
						FROM @ResponseLevels WHERE QAFlagID = 2 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 255)) AS AmberLevel,
			dbo.FNC_UnescapeXMLPath(
			substring((	SELECT '', '' + QuestionLevel
						FROM @ResponseLevels WHERE QAFlagID = 3 AND ResponseID = response.ResponseID
						FOR XML PATH('''')), 3, 255)) AS RedLevel						
	FROM (SELECT ResponseID FROM @ResponseMatches UNION SELECT ResponseID FROM @ResponseLevels) response
	GROUP BY ResponseID 
				
	RETURN 

END
' 
END

GO
/****** Object:  UserDefinedFunction [dbo].[FNC_User_ResponseList]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_User_ResponseList]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
BEGIN
execute dbo.sp_executesql @statement = N'-- =============================================
-- Author:		Markp
-- Create date: 17/08/2009
-- Description:	Returns list of responses that user can access
-- Version: 	11
-- =============================================
-- JT TFS 4194 - 14/08/2012 - When stock code is restricted, this now returns checks where no stock code is specified in the header.
-- JE TFS 4622 - returns QAFlagID based on ReportQAFlag config setting
-- JT TFS 5801 - performance improvements
-- JE TFS 5905 - updated to make use of Reporting Warehouse table
-- MP TFS 6031 - changed non-admin filter to ensure that temp checks are not included
-- MP TFS 6152 - fix to handle supplier site permissions
-- JT TFS 6689 - FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7612 - Void checks visibility is controlled by UserSetting for specific reports.
-- JE TFS 13-3570: added PRIMARY KEY to @Table_StockCode to improve performance
-- MP TFS 13-12783: fixed issue with ambiguous Location permission linking
-- =============================================
CREATE FUNCTION [dbo].[FNC_User_ResponseList]
(
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	ResponseStatus tinyint,
	QAFlagID int
)
AS
BEGIN

--declare @UserID int, @DateFrom datetime,	@DateTo datetime
--select @UserID = 836, @DateFrom = ''1 Jun 2013 00:00:00'',  @DateTo = ''21 May 2014 23:59:59''

	DECLARE @UnfilteredResponse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	--get date reporting warehouse table last built
	DECLARE @WarehouseBuildDate datetime
	SET @WarehouseBuildDate = (SELECT Updated FROM APP_Config WHERE OptionID = 78)

	DECLARE @UserSettingDisplayVoids bit 
	SET @UserSettingDisplayVoids = ISNULL(cast(dbo.FNC_GetUserSetting(@UserID, ''DisplayVoidChecks'') as bit), 0)

	-- Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserSettingDisplayVoids = 1 -- Include all recordstatuses when we set a report to do the same by setting the User Setting.
		SET @RecordStatus = 255 -- maximum possible. 
	ELSE IF @UserLevel = 1  --Admin can always see hidden/void checks.
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 100

	-- Permissions that this user has.
	DECLARE @PermissionLink TABLE (
		UserID int, SupplierID int, CustomerID int, LocationID int, StockCodeID int, 
		StockCodeSubGroup nvarchar(100), BusinessGroup nvarchar(100)
	)
	INSERT INTO	@PermissionLink (UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup)
	SELECT	UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup
	FROM	VW_PermissionLink PermissionLink
	WHERE	PermissionLink.PermissionAllow = 1 AND PermissionLink.PermissionType IN (101,102,103,104,105,106) AND UserID = @UserID

	DECLARE @Table_StockCode TABLE (
		StockCodeID int PRIMARY KEY
	)
	
	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code, and translate to StockCodeID
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		BusinessGroup IN (SELECT BusinessGroup FROM @PermissionLink WHERE BusinessGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @PermissionLink WHERE StockCodeSubGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeID IN (SELECT StockCodeID FROM @PermissionLink WHERE StockCodeID IS NOT NULL)

	BEGIN
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 AND @DateTo <= @WarehouseBuildDate --use warehouse table if built within specified date range
		BEGIN
			INSERT INTO	@UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
			SELECT		RPT_Warehouse_Response.ResponseID, QMS_Response.ResponseStatus, RPT_Warehouse_Response.QAFlagID --Actual or Override QAFlag setting checked when building warehouse table
			FROM		RPT_Warehouse_Response INNER JOIN
							QMS_Response ON RPT_Warehouse_Response.ResponseID = QMS_Response.ResponseID
			WHERE		(QMS_Response.RecordStatus < @RecordStatus) AND (RPT_Warehouse_Response.EndDate BETWEEN @DateFrom AND @DateTo)
			
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL)
				UPDATE	UnfilteredResponse 
				SET		LocationID = RPT_Warehouse_Response.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
				UPDATE	UnfilteredResponse
				SET		SupplierID = RPT_Warehouse_Response.SupplierID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
				UPDATE	UnfilteredResponse
				SET		CustomerID = PM_VW_StockCode.CustomerID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID INNER JOIN
							PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID

			IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
				UPDATE	UnfilteredResponse
				SET		StockCodeID = RPT_Warehouse_Response.StockCodeID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
		END
	ELSE --don''t use warehouse table
		BEGIN
			DECLARE @override bit 
			SET @override = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
			
			INSERT INTO @UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
			SELECT ResponseID, ResponseStatus,
					--check config value to use Actual or Override QAFlagID
					CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
			FROM	QMS_Response
			WHERE	(RecordStatus < @RecordStatus) AND (EndDate BETWEEN @DateFrom AND @DateTo)
			
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
				UPDATE	UnfilteredResponse
				SET		LocationID = Location.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
				UPDATE	UnfilteredResponse
				SET		LocationID = Location.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
				UPDATE	UnfilteredResponse
				SET		SupplierID = QMS_ResponseHeader.Header_SourceID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1)
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
				UPDATE	UnfilteredResponse
				SET		CustomerID = PM_VW_StockCode.CustomerID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID INNER JOIN
							PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID

			IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
				UPDATE	UnfilteredResponse
				SET		StockCodeID = HeaderStockCode.StockCodeID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID
		END
	END

	-- Only return responses which match the filter specified, or for which we do not have a filter.
	-- This means that admin users (any user with no permissions set) can see all responses.
	INSERT INTO @Response (ResponseID, ResponseStatus, QAFlagID)
	SELECT	ResponseID, ResponseStatus, QAFlagID
	FROM	@UnfilteredResponse UnfilteredResponse
	WHERE (	NOT EXISTS (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL)
		OR	LocationID IN (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
		OR	SupplierID IN (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL)
		OR	CustomerID IN (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL) )
	AND (	StockCodeID IS NULL -- Allow unlinked stockcode responses
		OR	NOT EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		OR	StockCodeID IN (SELECT StockCodeID FROM @Table_StockCode) )
	
	RETURN
END
' 
END

GO
/****** Object:  UserDefinedFunction [dbo].[FNC_Assessment_CustomerID]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_Assessment_CustomerID]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
BEGIN
execute dbo.sp_executesql @statement = N'-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 26/01/2015
-- Description:	Gets customer id for assessments (use with CROSS/OUTER APPLY)
-- Version:			1
-- =============================================
-- JT TFS 13-12788: Created
-- =============================================
CREATE FUNCTION [dbo].[FNC_Assessment_CustomerID] (
	@AssessmentID int
)
RETURNS TABLE 
AS
RETURN (	SELECT assessment.Spec_SourceID AS CustomerID
			FROM QMS_AssessmentSpecHeader as assessment
			INNER JOIN QMS_SpecificationHeader as spec ON assessment.SpecificationHeaderID = spec.SpecificationHeaderID
			WHERE assessment.AssessmentID = @AssessmentID
			AND Sys_SpecHeader = ''QAS_Customer'' 
			AND NULLIF(assessment.Spec_SourceID, 0) IS NOT NULL -- They do have a customer specified
			AND assessment.NotApplicable = 0
			AND assessment.RecordStatus < 201
			AND spec.RecordStatus < 201
)
' 
END

GO

PRINT 'Updating Stored Procedures...'
GO
/****** Object:  StoredProcedure [dbo].[QRY_Response_GetDUDays]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_Response_GetDUDays]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[QRY_Response_GetDUDays]
GO
/****** Object:  StoredProcedure [dbo].[QRY_QuestionPool_AdminList]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_QuestionPool_AdminList]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[QRY_QuestionPool_AdminList]
GO
/****** Object:  StoredProcedure [dbo].[QRY_Assessment_IsReadyForFinalSignOff]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_Assessment_IsReadyForFinalSignOff]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[QRY_Assessment_IsReadyForFinalSignOff]
GO
/****** Object:  StoredProcedure [dbo].[QRY_Assessment_AdminListSignOff]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_Assessment_AdminListSignOff]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[QRY_Assessment_AdminListSignOff]
GO

/****** Object:  StoredProcedure [dbo].[QRY_QuestionPool_AdminList]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_Import_PM_StockBatchItems]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[SYNC_Import_PM_StockBatchItems]
GO


/****** Object:  StoredProcedure [dbo].[QRY_QuestionPool_AdminList]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_Import_PM_StockBatchItems]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[SYNC_Import_PM_StockBatchItems] AS' 
END
GO
-- =============================================
-- Author:		jf
-- Create date: 11/2/15
-- Description:	Imort StockBatchItems
-- Version:		1
-- =============================================
-- jf 11/2/15 - added @TotalQuantity
-- =============================================
ALTER PROCEDURE [dbo].[SYNC_Import_PM_StockBatchItems] 
	@StockItemID int, 
	@StockBatchID INt, 
	@StockCodeID int, 
	@SupplierID int, 
	@UseByDate datetime, 
	@RecordStatus tinyint, 
	@Source tinyint, 
	@Updated datetime,  
	@GUID uniqueidentifier,
	@Created bigint,
	@RowVersion timestamp,
	@TotalQuantity decimal(18,2),

	@InsertCount int = 0 OUTPUT,
	@UpdateCount int = 0 OUTPUT,
	@ReceivedAnchor bigint = 0 OUTPUT
AS

IF NOT EXISTS(SELECT * FROM PM_StockBatchItems WHERE (StockItemID = @StockItemID))
BEGIN --Append record
	IF @RecordStatus < 201
	BEGIN
		INSERT INTO PM_StockBatchItems
						(StockBatchID, StockCodeID, SupplierID, UseByDate, RecordStatus, Source, Updated, GUID, TotalQuantity )
                      --(StockBatchID, BatchType, LocationID, Barcode, BatchRef, QAStatus, DateAdded, DateClosed, ExpiryDate, Closed, RecordStatus, GUID)
		VALUES     (@StockBatchID,@StockCodeID,@SupplierID,@UseByDate,@RecordStatus,@Source,@Updated,@GUID,@TotalQuantity)
		SET @InsertCount = @@rowcount
	END
END

ELSE
BEGIN --Update record
	UPDATE    PM_StockBatchItems
	SET              StockBatchID=@StockBatchID, StockCodeID=@StockCodeID, SupplierID=@SupplierID, UseByDate=@UseByDate, RecordStatus=@RecordStatus, Source=@Source, Updated=@Updated, TotalQuantity=@TotalQuantity
	WHERE     (StockItemID = @StockItemID)
	SET @UpdateCount = @@rowcount
END

IF CAST(@RowVersion AS bigint) > @Created
	SET @ReceivedAnchor = CAST(@RowVersion AS bigint)
ELSE
	SET @ReceivedAnchor = @Created

GO






/****** Object:  StoredProcedure [dbo].[QRY_Assessment_AdminListSignOff]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_Assessment_AdminListSignOff]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[QRY_Assessment_AdminListSignOff] AS' 
END
GO
-- =============================================
-- Author:		Markp
-- Create date: 26/10/2009
-- Description:	Gets list of assessments pending sign-off
-- Version:		3
-- =============================================
-- 4578 - populate new @AssessmentStatusID parameter in FNC_AssessmentQuestion_VersionChanges
-- JT TFS 13-12784: Also check status of "Partial Sign-off"
-- =============================================
alter PROCEDURE [dbo].[QRY_Assessment_AdminListSignOff] 
	@AssessmentTypeID int
AS

SELECT     CAST(0 AS bit) AS [Select], QMS_Assessment.AssessmentID, QMS_Assessment.AssessmentName, QMS_Assessment.Version, 
                      ISNULL(QMS_Assessment.Updated, QMS_Assessment.Inserted) AS Modified, QMS_User.FullName AS ModifiedBy, 
                      CurrentCheckOut.UserID AS CheckedOutUserID, CurrentCheckOut.FullName AS CheckedOutBy, QMS_Assessment.PreviousAssessmentID, 
                      PreviousAssessments.Version AS PreviousVersion, PreviousAssessments.AssessmentStatusID AS PreviousStatusID, 
                      AssessmentQuestion_VersionChanges.QuestionText AS [Question changes]
FROM         QMS_Assessment LEFT OUTER JOIN
                      QMS_User ON QMS_Assessment.UserID = QMS_User.UserID LEFT OUTER JOIN
                      dbo.FNC_AssessmentQuestion_VersionChanges(5) AS AssessmentQuestion_VersionChanges ON 
                      QMS_Assessment.AssessmentID = AssessmentQuestion_VersionChanges.AssessmentID LEFT OUTER JOIN
                          (SELECT     AssessmentID, AssessmentName, Version, Sys_AssessmentID, AssessmentStatusID
                            FROM          QMS_Assessment AS QMS_Assessment_Previous
                            WHERE      (RecordStatus < 201)) AS PreviousAssessments ON 
                      QMS_Assessment.PreviousAssessmentID = PreviousAssessments.AssessmentID LEFT OUTER JOIN
                          (SELECT     VW_RecordLock_Current.RowGUID, VW_RecordLock_Current.UserID, QMS_User_1.FullName
                            FROM          VW_RecordLock_Current INNER JOIN
                                                   QMS_User AS QMS_User_1 ON VW_RecordLock_Current.UserID = QMS_User_1.UserID
                            WHERE      (VW_RecordLock_Current.ObjectName = N'QMS_Assessment')) AS CurrentCheckOut ON 
                      QMS_Assessment.GUID = CurrentCheckOut.RowGUID
WHERE     (QMS_Assessment.RecordStatus < 201) AND (QMS_Assessment.AssessmentTypeID = @AssessmentTypeID) AND 
                      (QMS_Assessment.AssessmentStatusID IN (5, 7) )
ORDER BY QMS_Assessment.AssessmentName

GO
/****** Object:  StoredProcedure [dbo].[QRY_Assessment_IsReadyForFinalSignOff]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_Assessment_IsReadyForFinalSignOff]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[QRY_Assessment_IsReadyForFinalSignOff] AS' 
END
GO
-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 12/01/2015
-- Description:	Is the assessment ready for final sign off?
-- Version:		2
-- =============================================
-- JT TFS 13-11815: Created
-- JT TFS 13-11815: Remove criteria for current user to have already done a partial sign-off.
-- =============================================
alter PROCEDURE [dbo].[QRY_Assessment_IsReadyForFinalSignOff]
	@AssessmentID int,
	@UserID int
AS
BEGIN
	DECLARE @CurrentUserLevel int
	SELECT @CurrentUserLevel = dbo.FNC_GetUserLevel(@UserID)

	-- Administrators can always sign off.
	IF @CurrentUserLevel = 1
	BEGIN
		SELECT convert(bit, 1) as 'IsReady'
		RETURN
	END

	-- Do we need customer sign off, too?
	DECLARE @CustomerSpecHeaderID int
	SELECT @CustomerSpecHeaderID = SpecificationHeaderID 
	FROM QMS_SpecificationHeader 
	WHERE Sys_SpecHeader = 'QAS_Customer' 
	AND RecordStatus < 201

	-- They must have a customerID specified and have IsRequiredForSignOff set at userlevel level.
	DECLARE @IsCustomerSignOffRequired bit
	SELECT @IsCustomerSignOffRequired = 
		CASE WHEN EXISTS (SELECT * FROM APP_UserLevel WHERE IsRequiredForSignOff = 1 AND UserLevelID = 6)
			 AND EXISTS (
				SELECT * FROM QMS_AssessmentSpecHeader 
				WHERE SpecificationHeaderID = @CustomerSpecHeaderID
				AND AssessmentID = @AssessmentID
				AND NULLIF(Spec_SourceID, 0) IS NOT NULL -- They do have a customer specified
				AND NotApplicable = 0
			)
		THEN 1 ELSE 0 END

	-- Are there any levels which have not signed off yet?
	SELECT CASE WHEN EXISTS (
		SELECT * 
		FROM APP_UserLevel as userlevel
		WHERE userlevel.IsRequiredForSignOff = 1
		AND dbo.FNC_Assessment_IsSignedOffByUserLevel(@AssessmentID, userlevel.UserLevelID) = 0
		AND (userlevel.UserLevelID <> 6 OR @IsCustomerSignOffRequired = 1)
		AND userlevel.UserLevelID <> @CurrentUserLevel --assumption is that current userlevel is asking if they can sign off now.
	) THEN 0 ELSE 1 END as 'IsReady'

END

GO


/****** Object:  StoredProcedure [dbo].[QRY_QuestionPool_AdminList]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_QuestionPool_AdminList]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[QRY_QuestionPool_AdminList] AS' 
END
GO
-- =============================================
-- Author:		Markp
-- Create date: 02/07/2008
-- Description:	Gets list of questions for admin
-- Version:		4
-- =============================================
-- J.Toolan 11/10/2011: Added hideExternal parameter to prevent editing external questions.
-- JT TFS 4033: Added Section Columns
-- JT TFS 13-14079: Procedure seems to suffer from parameter sniffing. Removing direct links to parameters.
-- =============================================
alter PROCEDURE [dbo].[QRY_QuestionPool_AdminList] 
	@ShowSystem bit = 0,
	@hideExternal bit = 0 -- Hide imported questions from higher up the food chain (Active Quality Chain)
AS
BEGIN
	SET NOCOUNT ON 

	DECLARE @m_ShowSystem bit,
			@m_HideExternal bit
	SELECT @m_ShowSystem = @ShowSystem,
			@m_HideExternal = @hideExternal

	SET NOCOUNT OFF

	SELECT	QMS_QuestionPool.QuestionID, 
			QMS_QuestionPool.Question, 
			QMS_QuestionPool.IsQuestionGroup, 
			QMS_QuestionPool.DataTypeID,
			(	SELECT     DataType
				FROM          APP_DataType
				WHERE      (DataTypeID = QMS_QuestionPool.DataTypeID)) AS Lookup_DataTypeID, 
			QMS_QuestionPool.DataLookupID,
			(	SELECT     DataLookup
				FROM          APP_DataLookup
				WHERE      (DataLookupID = QMS_QuestionPool.DataLookupID)) AS Lookup_DataLookupID, 
			QMS_QuestionPool.CustomLookupID,
			(	SELECT     LookupName
				FROM          QMS_CustomLookup
				WHERE      (CustomLookupID = QMS_QuestionPool.CustomLookupID)) AS Lookup_CustomLookupID, 
			QMS_QuestionPool.QuestionLevelID,
			(	SELECT     QuestionLevel
				FROM          APP_QuestionLevel
				WHERE      (QuestionLevelID = QMS_QuestionPool.QuestionLevelID)) AS Lookup_QuestionLevelID, 
			QMS_QuestionPool.QuestionGroup,
			QMS_QuestionPool.QuestionSubGroup,
			QMS_QuestionPool.ShortGuidance, 
			QMS_QuestionPool.QuestionTag, 
			QMS_QuestionPool.AssessmentTypeID,
			(	SELECT     AssessmentType
				FROM          APP_AssessmentType
				WHERE      (AssessmentTypeID = QMS_QuestionPool.AssessmentTypeID)) AS Lookup_AssessmentTypeID, 
			QMS_QuestionPool.DefaultQuestion, 
			QMS_QuestionPool.CopyRepeatCheck, 
			QMS_QuestionPool.Hide, 
			ISNULL(ActiveAssessmentQuestion.QuestionCount, 0) AS ActiveQuestionCount, 
			CAST(CASE WHEN Sys_Question IS NULL THEN 0 ELSE 1 END AS bit) AS SystemQuestion
	FROM         QMS_QuestionPool 
	LEFT OUTER JOIN	(	SELECT		QMS_AssessmentQuestion.QuestionID, COUNT(QMS_AssessmentQuestion.AssessmentQuestionID) AS QuestionCount
						FROM        QMS_Assessment 
						INNER JOIN	APP_AssessmentStatus ON QMS_Assessment.AssessmentStatusID = APP_AssessmentStatus.AssessmentStatusID 
						INNER JOIN	QMS_AssessmentQuestion ON QMS_Assessment.AssessmentID = QMS_AssessmentQuestion.AssessmentID
						WHERE      (QMS_Assessment.RecordStatus < 201) AND (APP_AssessmentStatus.Active = 1) AND (QMS_AssessmentQuestion.RecordStatus < 201)
						GROUP BY QMS_AssessmentQuestion.QuestionID) AS ActiveAssessmentQuestion ON 
												  QMS_QuestionPool.QuestionID = ActiveAssessmentQuestion.QuestionID
	WHERE   (QMS_QuestionPool.RecordStatus < 201) 
	AND		(	QMS_QuestionPool.AssessmentTypeID IS NULL 
			OR
				NOT (QMS_QuestionPool.AssessmentTypeID IN
							  (SELECT     AssessmentTypeID
								FROM          APP_AssessmentType AS APP_AssessmentType_2
								WHERE      (Hide = 1)))) 
	AND		( (@m_ShowSystem = 1)		OR (QMS_QuestionPool.Sys_Question IS NULL) )
	AND     ( (@m_HideExternal = 0)	OR (dbo.FNC_Question_IsControlledExternally (QMS_QuestionPool.QuestionID) = 0) )
	ORDER BY QMS_QuestionPool.Question
END

GO
/****** Object:  StoredProcedure [dbo].[QRY_Response_GetDUDays]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_Response_GetDUDays]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[QRY_Response_GetDUDays] AS' 
END
GO
-- =============================================
-- Author:		Markp
-- Create date: 21/08/2009
-- Description:	Gets active assessments
-- Version:		2
-- =============================================
-- MP TFS 14406: Added UserID parameter to resolve timeout issue
-- =============================================
alter PROCEDURE [dbo].[QRY_Response_GetDUDays] 
	@ResponseID int,
	@UserID int
AS

SELECT QMS_AssessmentTarget.DUDays, QMS_Response.ResponseID
FROM         QMS_AssessmentTarget INNER JOIN
                      QMS_Response ON QMS_AssessmentTarget.AssessmentID = QMS_Response.AssessmentID INNER JOIN
                      dbo.FNC_ResponseHeader_StockCodeID_Selected(@UserID) AS FNC_ResponseHeader_StockCodeID_1 ON 
                      QMS_Response.ResponseID = FNC_ResponseHeader_StockCodeID_1.ResponseID AND 
                      QMS_AssessmentTarget.SourceObjectID = FNC_ResponseHeader_StockCodeID_1.StockCodeID
WHERE     (QMS_AssessmentTarget.RecordStatus < 201) AND (QMS_Response.ResponseID = @ResponseID)



GO


--PRINT 'Updating sync procedures...'

--PRINT 'Data migration tasks...'

PRINT 'Database updated successfully!'
GO
